/* TextWindow class allows printing of text.  The caret shows
 the current text insertion point, located at point (xPos, yPos). */!!

inherit(Window, #TextWindow,
#(
tmWidth     /* Width of font in pixels */
tmHeight    /* Height of font in pixels */
xPos        /* X coordinate of caret location */
yPos        /* Y coordinate of caret location */
xMax        /* Maximum number chars printable in line */
textMetrics /* A Struct with font information */
chStr       /* A String used in printChar */
), 2, nil) !!



now(TextWindowClass) !!

now(TextWindow) !!

/* Initialize a TextWindow.  Load the font data into textMetrics,
  set the text width and height instance variables, and home the caret. */
Def init(self | hdc)
{ textMetrics := new(Struct, 32);
  Call GetTextMetrics(hdc := Call GetDC(hWnd), textMetrics);
  tmWidth := asInt(wordAt(textMetrics, 10));
  tmHeight := asInt(wordAt(textMetrics, 8)) +
    asInt(wordAt(textMetrics, 0));
  Call ReleaseDC(hWnd, hdc);
  chStr := asciiz(
  " ");
  home(self);
} !!


/* Send cursor to home position.  Does not move the caret. */
Def home(self)
{ xPos := yPos := 0
} !!


/* Home the cursor, clear the screen. */
Def cls(self)
{	home(self);
	invalidate(self);
}!!

/* Translate xPos and return current x coordinate in pixels.  */
Def x(self)
{ ^xPos * tmWidth + 2
} !!

/* Translate yPos and return current y coordinate in pixels.  */
Def y(self)
{ ^yPos * tmHeight + 2
} !!


/* Move the caret to the current text insertion point. */
Def moveCaret(self)
{ ^Call SetCaretPos(x(self), y(self))
} !!

/* Display the TextWindow and calculate a new value for
  the maximum number of characters per line.  The val
  argument determines how the window will appear.  See the
  Actor manual, Guide to the Actor Classes, Window class, to
  see the various possible values and effects for val.  */
Def show(self, val )
{ show(self:Window, val);
  xMax := right(clientRect(self)) / tmWidth - 1
} !!


/* Clear the window if resized. */
Def WM_SIZE(self, wp, lp)
{	cls(self);
}!!

/* MS-Windows message indicating that self has just gotten
  the input focus.  When regaining focus, create a new caret. */
Def WM_SETFOCUS(self, wP, lP )
{ xMax := right(clientRect(self)) / tmWidth - 1;
  Call CreateCaret(hWnd, 0, 0, tmHeight);
  moveCaret(self);
  Call ShowCaret(hWnd);
  WM_SETFOCUS(self:Window, wP, lP);
} !!


/* MS-Windows notification that self is just about to lose
  the input focus.  Hide caret first. */
Def WM_KILLFOCUS(self, wP, lP)
{
Call HideCaret(hWnd);
Call DestroyCaret();
^0
} !!


/* Draw a character in the window at current position.
  Go the next line if character is a CR. */
Def drawChar(self, aChar | hdc)
{
  if aChar == CR
  then ^eol(self)
  endif;
  hdc := Call GetDC(hWnd);
  Call SetBkMode(hdc, OPAQUE);
  chStr[0] := max(aChar, ' ');
  Call TextOut(hdc, x(self), y(self), chStr, 1);
  Call ReleaseDC(hWnd, hdc);
  ^aChar
} !!

/* Print a character in window at the text insertion point.
  Skip line if necessary. */
Def printChar(self, aChar)
{
  if xPos >= xMax
  then  eol(self);
  endif;
  drawChar(self, aChar);
  xPos := xPos + 1;
  ^aChar
} !!


/* Draw a string in the TextWindow. */
Def drawString(self, aStr | hdc)
{	Call SetBkMode(hdc := Call GetDC(hWnd), OPAQUE);
	Call TextOut(hdc, x(self), y(self), aStr, size(aStr));
	Call	ReleaseDC(hWnd, hdc);
	^aStr
}!!

/* Print the string, on new line if necessary. */
Def printString(self, aStr)
{	if	xPos + size(aStr) > xMax
	then  eol(self);
	endif;
	drawString(self, aStr);
	xPos := xPos + size(aStr);
	^aStr;
}!!

/* Backspace once.  Cannot backup to the previous line. */
Def bs(self)
{ xPos := max(xPos-1, 0);
  ^drawChar(self, ' ');
} !!

/* End of line--CR.  If near the bottom of the window,
 scroll window up on line. */
Def eol(self | rect ht)
{ xPos := 0;
  yPos := yPos + 1;
  ht := tmHeight;
  if ht * (yPos+2) > bottom(clientRect(self))
  then Call ScrollWindow(hWnd, 0, negate(ht), 0, 0);
    rect := clientRect(self);
    setTop(rect, bottom(rect)-(ht));
    Call ValidateRect(hWnd, rect);
    yPos := yPos - 1;
  endif;
} !!







