/* The Browser class creates and manages Browser windows, the
  part of Actor which manages all the source code for the Actor
  system.  Specifically, this class manages the Browser listboxes,
  edit window, and all other Browser functions. */   !!

inherit(ToolWindow, #Browser,
#(mode /* Determines whether class or object methods are displayed */
  selClass /* The currently selected class */
  classCol /* A collection of classes */
  comment /* The class's comment string */
  inherit /* The class's inherit message string */
  alpha /* Alphabetic/hierarchial class listing flag */
        ), 2, nil) !!


now(BrowserClass) !!

now(Browser) !!

/* Close the receiver browser and remove from the master list of Browsers. */
 Def WM_CLOSE(self, wP, lP )
{
  if doDirtyWork(self)
  then Call DestroyWindow(hWnd);
    remove(TheApp.workspace.browsers, self);
  endif;
  ^0
}!!


/* Open the selected class file in the WORK directory, if the
  class is in dirtyClasses.  Otherwise, open the class file in
  the CLASSES directory.  Return the opened file. */
Def openClassFile(self, class | rFile)
{ rFile := new(SourceFile);
  if openClass(rFile, class)
  then ^rFile;
  endif;
  ^nil;
} !!


/* Save the new text for current method into the source file.
  The first argument, text, is the method text.  The second,
  fSym, is the symbol with the name of the method, e.g. #print.
  First an attempt is made to replace previous method text,
  otherwise, method is simply added to the file.  */
Def  saveMethText(self, text, fSym | wFile, rFile,pnt, fName)
{ rFile := openClassFile(self, selClass);
  wFile := delReplMethod(rFile, text, fSym, true);
  select
    case mode = BR_OMETH and not(wFile)
    is   wFile := addObjectMeth(rFile, text)
    endCase
    case mode = BR_CMETH and not(wFile)
    is   wFile := addClassMeth(rFile, text)
    endCase
  endSelect;
  close(rFile);
  fName := condDelCFile(rFile, selClass);
  reName(wFile, fName);
  add(DirtyClasses,   selClass.name);
} !!

/* Compile the method code in the edit window. */
Def accept(self | fSym)
{
  if size(Parser.lex.collection :=
    asString(ew.workText)) > 0
  then showWaitCurs(self);
    CStream.collection := CompileBuf;
    Parser.lex.createFlag := 1;
    if mode = BR_CMETH
    then now(class(selClass));
    else now(selClass);
    endif;
    if not(Compiler.curClass.methods)
    then Compiler.curClass.methods :=
      new(MethodDictionary,2);
    endif;
    selectAll(ew);
    parse(Parser, nil);
    if (fSym := Compiler.curFunc)
    then saveMethText(self, ew.workText,
      fSym);
      initEditParms(ew);
      ew.dirty := nil;
      loadMethods(self);
      selectString(lb2, fSym);
    endif;
    setFocus(ew);
    showOldCurs(self);
  endif;
  invalidate(ew);
} !!


/* Handle the menu choices (Accept, Reformat, Delete method, etc). */
Def  doMenuChoice(self, wP)
{
  select
    case wP == BR_REFORM
    is reform(ew);
    endCase
    case wP == BR_DELCL
    is delSelClass(self);
    endCase
    case wP == BR_DELME
    is delSelMethod(self);
    endCase
    case wP == BR_ACCEPT and selClass
    is accept(self);
    endCase
    case wP >= TEMP_DO and wP <=
      TEMP_NMETH and selClass
    is mergeTemplate(ew, tempStr(ew, wP));
    endCase
    case wP == INSP_DOIT
    is  doLine(ew);
    endCase
    case wP == INSP_ISEL
    is inspectIt(ew);
    endCase
  default options(self, wP);
  endSelect;
  ^0
} !!



/* Remove the selected class and all descendants (if any) from Actor.  */
 Def delSelClass(self | classes, clStr, should)
{ showWaitCurs(self);
  classes := descendants(selClass);
  clStr := "";
  do(classes,
  {using(aCl) clStr := clStr + aCl.name + " ";
  });
  if IDYES == new(ErrorBox, self, clStr,
    "Delete these Classes?", MB_YESNO + MB_ICONEXCLAMATION)
  then
    if classCol
    then remove(classCol,   selClass)
    endif;
    if   selClass.name in DirtyClasses
    then remove(DirtyClasses, selClass.name);
    endif;
    selClass := nil;
    initCache();
    do(classes,
    {using(aCl) remove(Actor, aCl.name)
    });
    resetClassMenu(self);
    fillClassList(self);
    clearList(lb2);
    setText(self, "Browser");
  endif;
  showOldCurs(self);
}!!



/* Remove the selected method from the currently selected class.  */
Def delSelMethod(self | method rFile wFile fName)
{  method :=
  asSymbol(getSelString(lb2));
  remove(selClass.methods, method);
  initCache();
  rFile := openClassFile(self, selClass);
  if (wFile := deleteMethod(rFile, method))
  then showWaitCurs(self);
    fName := condDelCFile(rFile, selClass);
    reName(wFile, fName);
    add(DirtyClasses, selClass.name);
    showOldCurs(self);
  endif;
  close(rFile);
  initWorkText(ew);
  loadMethods(self);
  cls(ew);
  initEditParms(ew);
  setFocus(ew)
} !!


/* Fill the class list box from classCol (if there is one).
  If there isn't, fill the list box either alphabetically
  or by class hierarchy (depending on value of Browser's
  instance variable alpha). */
Def fillClassList(self | aSet)
{       select
        case classCol
        is fill(lb1, classCol, selClass)
        endCase
        case alpha
        is fill(lb1, asSortedCollection(classes(Actor)), selClass)
        endCase
        default fill(lb1, nil, selClass)
        endSelect
} !!

/* Handle all of the Option menu's choices (About the class, etc.). */
Def options(self, wP)
{
  select
    case wP == BR_CMETH and mode == BR_OMETH
    is      cMethods(self)
    endCase
    case wP == BR_OMETH and mode == BR_CMETH
    is      oMethods(self)
    endCase
    case wP == BR_CABOUT
    is aboutCl(self)
    endCase
    case wP == BR_CDES
    is makeDescendant(self)
    endCase
    case wP == BR_ZOOM
    is zoomEdit(self)
    endCase
    case (wP == BR_ALPH      and not(alpha)) or (wP == BR_HIER
      and alpha)
    is showWaitCurs(self);
      if alpha        and not(classCol)
      then unCheck(self, BR_ALPH);
        check(self, BR_HIER)
      else check(self, BR_ALPH);
        unCheck(self, BR_HIER)
      endif;
      alpha := not(alpha);
      fillClassList(self);
      showOldCurs(self)
    endCase
  default WM_COMMAND(ew, wP, 0);
  endSelect
} !!


/* Switch mode and display class methods rather than object methods.  */
Def  cMethods(self)
{       mode := BR_CMETH;
        check(self, BR_CMETH);  unCheck(self, BR_OMETH);
        initWorkText(ew);
        initEditParms(ew);
        repaint(ew);
        loadMethods(self); setFocus(ew)
}!!

/* Switch mode and display object methods rather than class methods.  */
Def  oMethods(self)
{       mode := BR_OMETH;
        check(self, BR_OMETH);  unCheck(self, BR_CMETH);
        initWorkText(ew);
        initEditParms(ew);
        repaint(ew);
        loadMethods(self);  setFocus(ew)
}!!

/* Change zoom state and update window accordingly. */
Def zoomEdit(self)
{       if zoom
        then unCheck(self, BR_ZOOM)
        else check(self, BR_ZOOM);
                setLastSel(lb1); setLastSel(lb2)
        endif;
        zoom := not(zoom);
        sizeKids(self); setFocus(ew);
        invalidate(self);
} !!

/* Setup for editing a newly selected class. */
 Def  initClassEdit(self)
{ setText(self,
  "Browser: " + selClass.name);
  initWorkText(ew);
  loadMethods(self);
  cls(ew);
  initEditParms(ew);
  setFocus(ew)
}!!


/* Create default class comment if none given.  Return
  the repaired array (in ClassDialog format).*/
 Def  fixArray(self, array)
{
  if limit(leftJustify(array[1])) = 0
  then array[1] := "/* " + array[0] +
    " class definition */";
  endif;
  ^array
}!!

/* Update the specified class file with the specified ClassDialog array.
  Only replace as many chunks as limit indicates. */
 Def updateCFile(self, class, array, limit | rFile)
{  rFile := openClassFile(self, class);
  updateClassFile(rFile, fixArray(self, array), limit);
}!!


/* Run the errorBox asking if classes should be recompiled
  and return a boolean flag according to the user's answer. */
 Def compClDia(self, clColl | cStr)
{cStr := "";
  do(clColl,
  {using(clSym) cStr := cStr + clSym + " ";
  });
  if IDYES = new(ErrorBox, self, cStr,
    "Recompile these classes?", MB_YESNO + MB_ICONEXCLAMATION)
  then ^true;
  endif;
  ^nil
}!!

/* Recompile the source code for all the classes
  in the specified collection of classes. */
 Def recompClasses(self, clColl | fileName)
{ do(clColl,
  {using(aCl)
    if aCl in DirtyClasses
    then fileName := "work\"
    else fileName := "classes\"
    endif;
    load(fileName + subString(aCl, 0, 8) +  ".cls");
  });
}!!

/* The ClassDialog returns an array with eight elements:
        array[0] is the class name
        array[1] is the comment
        array[2] is the parsable inherit string
        array[3] is "now(classClass)"
        array[4] is the ivar string
        array[5] is the format
        array[6] is isIdx
        array[7] is the ancestor name  */ !!

/* Present the "About the class" dialog for the selected
  class and handle the result. */
Def aboutCl(self | array, clColl)
{
  if not(loadClassInfo(self))
  then ^nil;
  endif;
  if array := new(ClassDialog, self,
    selClass, selClass.ancestor)
  then  showWaitCurs(self);
    clColl :=
      shouldCompile(Actor[array[7]],
      selClass.name, asLiteral(array[4]),
      array[5], array[6]);
    if clColl
    /* classes to compile */
    then
      if compClDia(self, clColl)
      /* should we? */
      then updateCFile(self, selClass,
        array, 3);
        recompClasses(self, clColl);
        selClass := Actor[selClass.name];
        fillClassList(self);
      else showOldCurs(self);
        ^invalidate(self);
      endif;
    else
      if subString(inherit, 0, 2) =
        "/*"
      then updateCFile(self, selClass,
        array, 1);
      else updateCFile(self, selClass,
        array, 2);
      endif;
    endif;
    add(DirtyClasses, selClass.name);
    loadClassInfo(self);
    showOldCurs(self);
  endif;
  invalidate(self);
} !!


/* Put up a new ClassDialog box for the possible new
  descendant.  If the descendant doesn't already exist, then
  the new class is created along with its source file.
  If it does exist, an error dialog is shown.  */
Def  makeDescendant(self | array, clColl)
{
  if array := new(ClassDialog, self, nil, selClass)
  then
    if Actor[array[0]]    /* Class already exists */
    then errorBox("Class Name Error", array[0] +
      " already exists.  Use About Class dialog.");
    else showWaitCurs(self);
      parse(array[2], nil);
      makeClassFile(new(SourceFile), fixArray(self, array));
      selClass := Actor[array[0]];
      if classCol
      then add(classCol, selClass)
      endif;
      fillClassList(self);
      add(DirtyClasses, selClass.name);
      initClassEdit(self);
      showOldCurs(self);
    endif;
  endif;
  invalidate(self);
}!!


/*  Reset the menu bar if a class is selected. */
Def   resetClassMenu(self)
{       if selClass
        then enable(self, BR_REFORM);
                enable(self, BR_CABOUT); enable(self, BR_CDES);
                enable(self, BR_DELCL);
        else gray(self, BR_REFORM);
                gray(self, BR_CABOUT); gray(self, BR_CDES);
                gray(self, BR_DELCL);
        endif;
} !!

/* Load the class information into Browser's instance
  variables (inherit and comment) from the class's source file. */
 Def loadClassInfo(self | rFile)
{
  if rFile := openClassFile(self, selClass)
  then comment := leftJustify(readChunk(rFile));
    inherit := leftJustify(readChunk(rFile));
    close(rFile);
    ^true;
  else ^nil;
  endif;
}!!

/* Load and return the text for the selected method.
  If the source code for the selected method isn't found,
  return nil. */
 Def  loadSelMethod(self | selMeth, rFile)
{ selMeth := getSelString(lb2);
  rFile := openClassFile(self, selClass);
  ^loadMethText(rFile, selMeth);
}!!


/* Show the Dirty Work (Accept, Abandon, etc.) dialog box
  if the text in the Browser's edit window has changed.  If
  the dialog box has to be shown, the various responses are
  handled.  */
 Def doDirtyWork(self | ans)
{
  if not(ew.dirty)
  then ^0
  endif;
  ans := new(ModalDialog, DW_BOX, self);
  select
    case ans = 0
    is ^nil
    endCase
    case ans == DW_ACC and selClass
    is accept(self)
    endCase
    case ans == DW_CTC
    is selectAll(ew);
      xCut(ew)
    endCase
    case ans == DW_ABA
    is initWorkText(ew);
    endCase
  endSelect;
  ew.dirty := nil;
  ^0
}!!


/* Load the method listbox with the class's methods
  if there is a selected class. */
Def   loadMethods(self | cl)
{		if not(selClass)
		then ^nil
		endif;
	
		cl := if mode == BR_CMETH
			then class(selClass)
			else selClass
			endif;

		clearList(lb2);
		sendMessage(lb2, WM_SETREDRAW, 0, 0);
		keysDo(cl.methods,
    	{using(meth)
   		addString(lb2, meth)
    	});
  		sendMessage(lb2, WM_SETREDRAW, 1, 0);
  		Call InvalidateRect(lb2.hCntl, 0, 0);
} !!

/* Handle the various Browser events. */
Def    WM_COMMAND(self, wP, lP | rFile)
{
  select
    case wP == CL_ID and high(lP) = LBN_SELCHANGE
      and getSelString(lb1)
    is
      if doDirtyWork(self)
      then  selClass := getSelClass(lb1);
        resetClassMenu(self);
        initClassEdit(self);
      else setLastSel(lb1);
      endif;
    endCase
    case wP == ML_ID and high(lP) = LBN_SELCHANGE and
      getSelString(lb2)
    is
      if doDirtyWork(self)
      then showWaitCurs(self);
        copyMethod(ew, loadSelMethod(self));
        reform(ew);
        enable(self, BR_DELME);
        setFocus(ew);
        showOldCurs(self)
      else setLastSel(lb2)
      endif;
    endCase
    case lP = 0
    is   doMenuChoice(self, wP)
    endCase
  endSelect;
  ^1
}!!


/* Doesn't do anything except override ancestor's method.  */
Def   paint(self, hDc)
{ } !!

/* Start up a Browser.  If clsCol is a collection of classes,
  only those classes are loaded.  If clsCol is nil (the default
  when you pick "Browse!" from a WorkSpace menu), all classes are
  loaded.  If the second argument, class, is not nil, then it
  will be the currently selected class when the Browser appears.
  Otherwise, the Browser will appear with no currently selected
  class.  */
Def   start(self, clsCol, class)
{ show(self, 1);
  showWaitCurs(self);
  selClass := class;
  classCol := clsCol;  /* if nil, all classes */
 newSize := oldSize := asPoint(longAt(clientRect(self),
  4));
  lb1 := new(ClassList, CL_ID, self);  /* class list box */
 lb2 := new(ListBox, ML_ID, self);  /* method list box */
 ew := new(BrowEdit, 0, self);
  sizeKids(self);
  check(self, BR_HIER);  /* initially hier.*/
 fillClassList(self);
  show(lb1, 1);
  show(lb2, 1);
  show(ew, 1);
  setFocus(ew);
  oMethods(self);
  showOldCurs(self);
}!!

setName(VImage, "actor.ima") !!

