/* This is the parent class of all edit-style windows.
  It supports text editing and cutting/pasting.  A new window
  class is registered with MS-Windows on start-up.  The caret
  for all EditWindow objects is the standard text-editing
  I-beam. */    !!

inherit(TextWindow, #EditWindow,
#(dragDC  /* The handle to a display context used for selecting text */
pOrigin   /* The point where draggging starts */
oldX      /* The previous x value while dragging */
dragLine  /* The current line for dragging */
workText  /* A TextCollection containing the text of the window */
topLine   /* The index into workText of the line at top of window */
caretVis  /* Boolean flag controlling visibility of caret */
dirty     /* Boolean flag--true if text has been changed */
startChar /* Starting character of highlighted range of text */
startLine /* Starting line of highlighted range of text */
endChar   /* Ending character of highlighted range of text */
endLine   /* Ending line of highlighted range of text */ 
), nil, nil) !!


add(Constants, #EditWClass, static(asciiz("EditWindow")))!!

now(EditWindowClass) !!

/* Return static string for this window class name ("EditWindow"). */
Def wndClass(self)
{ ^EditWClass;
}!!

/* Register all edit windows with the I-beam caret style. */
Def register(self | wc)
{ wc := newWClass(self,
  wndClass(self), "work");
  putWord(wc, Call LoadCursor(0,
    IDC_IBEAM), 14);
  Call RegisterClass(wc);
}!!


now(EditWindow) !!

/* Modify the style for edit windows--add a vertical scroll bar. 
  Permits edit windows to be tile, popup, or child style. */
Def  create(self, par, wName, rect, style)
{	create(self:Window, par, wName, rect,
		style + WS_VSCROLL)
} !!

/* Initialize the edit window. */
Def init(self)
{ init(self:TextWindow);
  initWorkText(self);
  initEditParms(self);
} !!


/* Initialize the editing parameters (home the caret, etc.). */
Def  initEditParms(self)
{ home(self);
  moveCaret(self);
  topLine := 0;
  initSelParms(self);
  setScrollPos(self);
} !!


/* Enable window for input and set the focus if window
 doewsn't already it. */
Def setFocus(self)
{ if Call GetFocus( ) <> hWnd
  then Call SetFocus(hWnd)
  endif;
} !!


/* Return true flag for error insertion routines. */
Def isEditable(self)
{ ^true
}!!

/* Recalculate xMax to max characters per line for formatting text.  */
Def WM_SIZE(self, wP, lP)
{ xMax := asInt(low(lP)) / tmWidth - 1;
  moveCaret(self);
  showCaret(self);
  ^0
} !!

/* Dispatch menu choices, accelerators.  */
Def WM_COMMAND(self, wP, lP)
{
  select
    case wP == EDIT_CUT
    is
      if not(xCut(self)) and high(lP) = 1
      then delChar(self);
      endif;
    endCase
    case wP == EDIT_COPY
    is xCopy(self)
    endCase
    case wP == EDIT_PASTE
    is xPaste(self);
      Call HiliteMenuItem(hWnd, hMenu,
        EDIT_PASTE, 0);
    endCase
    case wP == EDIT_CLEAR
    is xClear(self)
    endCase
    case high(lP) <> 1
    is ^0
    endCase
    case wP == EDIT_SELALL
    is selectAll(self);
      repaint(self);
    endCase
    case wP == EDIT_PRIOR
    is WM_VSCROLL(self, SB_PAGEUP, 0);
    endCase
    case wP == EDIT_NEXT
    is WM_VSCROLL(self, SB_PAGEDOWN, 0);
    endCase
    case wP == EDIT_TAB
    is WM_CHAR(self, 32, 0);
      if xPos bitAnd 1 > 0
      then WM_CHAR(self, 32, 0);
      endif;
    endCase
  default arrows(self, wP)
  endSelect;
  ^0
} !!


/* Respond to arrow keys (accelerators).
  Not implemented at this time.  wP is
  ID of accelerator. */
Def  arrows(self, wP | dx, dy)
{  ^ 0
}!!

/* Delete the character to the right of caret. */
 Def  delChar(self)
{
  select
    case xPos < limit(workText[startLine])
    is deleteChar(workText, startLine, xPos);
      printLine(self, startLine); dirty := true;
    endCase
    case startLine < size(workText) - 1
    is deleteText(workText, startLine, xPos, startLine+1,
      0); dirty := true;
      invalidate(self);
      moveCaret(self);
      showCaret(self);
    endCase
  endSelect;
}!!

/* Prepare window for input and output, show selected text. */
Def WM_SETFOCUS(self, wP, lP )
{	Call EnableWindow(hWnd, 1);
	xMax := right(clientRect(self)) / tmWidth - 1;
	Call CreateCaret(hWnd, 0, 0, tmHeight-2);
	moveCaret(self); caretVis := nil;
	invertSelText(self);
	showCaret(self);
	ThePort := self; ^0
}!!

/* When losing focus, de-select text visually, and
  then hide and destroy the caret. */
Def WM_KILLFOCUS(self, wP, lP)
{	invertSelText(self);  hideCaret(self);
	Call DestroyCaret();
  ^0
}!!


/* Initialize the workText instance variable,
  add one zero-length string. */
Def initWorkText(self)
{	workText := new(TextCollection, 8);
	add(workText, "")
} !!

/* Hide the caret if it is visible and switch caretVis flag. */
Def hideCaret(self)
{	if caretVis
	then Call HideCaret(hWnd); caretVis := nil
	endif;
} !!

/* Show the caret if it is hidden and switch caretVis flag. */
Def showCaret(self)
{	if not(caretVis or isSelText(self))
	then Call ShowCaret(hWnd); caretVis := true;
	endif;
} !!

/* Adjust topLine if near the top or bottom of window, return flag
  that topLine was adjusted. */
Def   resetTop(self)
{	if (startLine-topLine)+3 > visLines(self)
	then topLine := min(size(workText)-1, topLine+3); ^true
	endif;
	if startLine-topLine < 0
	then topLine := topLine - 1; ^true
	endif;
	^nil
} !!

/* Handle the inputted character, return true if aChar is a CR. 
 Delete selected text first. */
Def charInput(self, aChar | line, isDel)
{	if not(isDel := deleteSelText(self))
  	then endChar := startChar := xPos; endLine := startLine
	endif;
   select
	case isPrintable(aChar)
	is   insertString(workText, asString(aChar), startLine, startChar);
		printLine(self, startLine);
		xPos := xPos + 1
	endCase

	case not(isDel) and aChar == BS and startChar > 0
	is	xPos := xPos - 1;
		deleteChar(workText, startLine, xPos);
		printLine(self, startLine)
	endCase

	case not(isDel) and aChar==BS and startChar==0 and startLine>0	
	is	startLine := startLine - 1;
		startChar := size(workText[startLine]);
		deleteSelText(self)
	endCase

	case aChar == CR	 and xPos > 0
	is   line := workText[startLine];
	     workText[startLine] := subString(line, 0, xPos);
	     insert(workText, subString(line, xPos, size(line)),
		startLine+1)
	endCase

	case aChar == CR and xPos == 0
	is   insert(workText, new(String, 0), startLine)
	endCase
 endSelect;
	^(aChar = CR)
}!!

/* Process MS-Window's character input message. */
Def WM_CHAR(self, wP, lP | rect)
{	hideCaret(self);
	if charInput(self, asChar(wP))
	then startLine := endLine := startLine +1;
	     if resetTop(self)
	     then repaint(self);
	     else rect := copy(clientRect(self));
		setTop(rect, y(self));
		Call InvalidateRect(hWnd, rect, 1);
		Call UpdateWindow(hWnd);
	     endif;	
	     yPos := startLine-topLine; xPos := 0;
        endif; 
   dirty := true;
	startChar := endChar := xPos;
	moveCaret(self); showCaret(self);
	^0
}!!

/* Redraw the workText from topLine down,
  preserve xPos and yPos.  If window has focus,
  show selected text.*/
Def   paint(self, hdc | aStr, xH, yH)
{ xH := xPos;
  yH := yPos;
  home(self);
  Call SetBkMode(hdc, OPAQUE);
  if noScroll(self)
  then topLine := 0;
    yH := startLine;
  endif;
  do(over(topLine, min(size(workText), 1 
    + topLine + visLines(self))),
  {using(idx)    aStr := workText[idx];
    Call TextOut(hdc, x(self), y(self),   
      aStr, size(aStr));
    xPos := 0;
    yPos := yPos + 1;
  });
  xPos := xH;
  yPos := yH;
  setScrollPos(self);
  if Call GetFocus( ) == hWnd
  then dragDC := hdc;
    invSelTxt(self);
  endif;
}!!


/* Print the given line, preserve xPos and yPos.  Obtain
  own display context. */
Def printLine(self, line | xH, yH, hdc, aStr)
{	if line < size(workText)
	then	xH := xPos;   yH := yPos;
		xPos := 0; yPos := line - topLine;
		aStr := workText[line] + " ";
		hdc := getContext(self);
		Call SetBkMode(hdc, OPAQUE);
		Call TextOut(hdc, x(self), y(self), aStr, size(aStr));
		xPos := xH; yPos := yH;
		releaseContext(self, hdc);
		endif;
} !!

/* Set the Clipboard to the specified text. */
Def setClipText(self, text | hnd)
{
  if Call OpenClipboard(hWnd) <> 0
  then hnd := asHandle(asciiz(text));
    Call SetClipboardData(CF_TEXT, hnd);
    ^Call CloseClipboard( );
  endif;
  ^nil
} !!


/* Return the text string from the Clipboard. */
Def getClipText(self | hStr, aStr)
{	if Call OpenClipboard(hWnd) <> 0
	then hStr := Call GetClipboardData(CF_TEXT);
	     aStr := getText(hStr);
	     Call CloseClipboard( );
	     ^removeNulls(aStr);
	endif;
	Call CloseClipboard( );
   ^nil
} !!

/* Cut the selected text to the clipboard. */
Def   xCut(self)
{	if isSelText(self)
	then setClipText(self, getSelText(self));
	     deleteSelText(self);
	     moveCaret(self); showCaret(self);	^1
	endif;  ^nil
} !!

/* Copy the selected text to the clipboard. */
Def   xCopy(self)
{	if isSelText(self)
	then setClipText(self, getSelText(self));
  ^true
	endif;  ^nil;
} !!

/* Paste the clipboard text to the EditWindow
  at current insertion point. */
Def   xPaste(self | aStr, aPnt)
{	deleteSelText(self);
	aStr := getClipText(self);
	aPnt := insertText(workText, aStr, startLine, startChar);
	endChar := aPnt.x; endLine := aPnt.y;
	repaint(self)
} !!

/* Clear the selected text. */
Def   xClear(self)
{	if deleteSelText(self)
	then moveCaret(self); showCaret(self);
   ^true
	endif;
  ^nil
} !!

/* Return the number of visible text lines in window. */
Def visLines(self)
{	^(height(clientRect(self))/tmHeight)
} !!

/* Decide if text doesn't need to be scrolled,
  i.e., if worktext all fits in window. */
Def noScroll(self)
{	^height(clientRect(self))/tmHeight > size(workText)
} !!

/* Set scroll bar position, avoiding divide by 0. */
Def setScrollPos(self)
{	Call SetScrollPos(hWnd, SB_VERT,
		 (100*topLine)/max(1, size(workText)), 1)
} !!

/* Respond to MS-Window's vertical scrolling message. 
  wP tells what kind of scrolling request has been made. */
Def WM_VSCROLL(self, wP, lP | oldTop)
{	oldTop := topLine;
  select
	case noScroll(self)
	  is ^0
	endCase
	case wP == SB_LINEDOWN	and topLine < size(workText) - 1
	  is	topLine := topLine + 1;
		Call ScrollWindow(hWnd, 0, negate(tmHeight),
			0, 0);
		Call UpdateWindow(hWnd);
	        setScrollPos(self);
	endCase
	case wP == SB_PAGEDOWN
	is	topLine := min(topLine + visLines(self), size(workText)-1);
		setScrollPos(self); repaint(self);
	endCase
	case  wP == SB_LINEUP and topLine > 0
	is	topLine := topLine - 1;
		Call ScrollWindow(hWnd, 0, tmHeight,
			0, 0);
		Call UpdateWindow(hWnd);
		setScrollPos(self);
	endCase
	case wP == SB_PAGEUP
	is	topLine := max(0, topLine - visLines(self));
		setScrollPos(self);  repaint(self);
	endCase
	case  wP == SB_THUMBPOSITION
	is	invertSelText(self);
		topLine := asInt(min((size(workText)*low(lP))/100,
		size(workText)-1));
		setScrollPos(self);  repaint(self);
	endCase
  endSelect;
	yPos := yPos + (oldTop - topLine);
	moveCaret(self); showCaret(self);
	^1
} !!

/* Return the selected text as a string suitable for clipboard,
  with a CR_LF between each line. */
Def getSelText(self)
{	if isSelText(self)
	then ^subText(workText, startLine, startChar, endLine, endChar)
	endif;
	^nil
} !!

/* Delete the selected text from the TextCollection, workText,
  and reset selection parameters. */
Def deleteSelText(self)
{	if isSelText(self)
	then deleteText(workText, startLine, startChar, endLine, endChar);
		endLine := startLine;
		xPos := endChar := startChar;
		resetTop(self);
		yPos := startLine - topLine;
		repaint(self);
		^0
	endif;
   dirty := true;
	^nil
} !!

/* Return true if there is selected text. */
Def isSelText(self)
{	^(startChar <> endChar or startLine <> endLine)
} !!

/* Set cursor position (xPos, yPos)
  according to the specified point. */
Def setCurPos(self, aPnt | yt)
{ yt := max(0, aPnt.y)/tmHeight;
  yPos := min((size(workText)-1) - topLine, yt);
  xPos := size(workText[yPos+topLine]);
  if yPos >= yt
  then xPos := max(min(xPos, aPnt.x/tmWidth),0)
  endif;
} !!



/* Show a 0-length line as selected by inverting a 2-pixel wide
  strip at left. Requires a valid dragDC. */
Def   selNulLine(self, yo)
{	Call PatBlt(dragDC, 2, yo, 1,	tmHeight-2, BLACKNESS)
} !!

/* Inverts the rectangle starting at xo and yo, width.
  Height is character height - 2.  Assumes a valid dragDC. */
Def   invertLine(self, xo, yo, width | wid)
{
  if xo == 2 and width == 0
  then wid := 1
  else wid := width
  endif;
  Call PatBlt(dragDC, xo, yo, wid, tmHeight-2, DSTINVERT);
} !!


/* Invert the selected text.  Assumes a valid dragDC.  */
Def  invSelTxt(self | width, ht)
{	if isSelText(self) and (not(endLine < topLine
		or startLine > topLine + visLines(self)))
	then	width := tmWidth; ht := tmHeight;
		Call SelectObject(dragDC, Call GetStockObject(BLACK_BRUSH));
		if startLine == endLine
		then invertLine(self, (startChar * width) + 2,
		     (startLine-topLine) * ht + 2,
		     (endChar-startChar) * width);
		else invertLine(self, (startChar * width) + 2,
		     (startLine-topLine) * ht + 2,
		     (size(workText[startLine]) - startChar) * width);
	             do(over(startLine+1, endLine),
			{using(line)  invertLine(self, 2,
			(line-topLine) * ht + 2,
			size(workText[line]) * width)
			});
		     invertLine(self, 2, (endLine-topLine) * ht + 2,
			endChar * width);
		endif; ^0;
	endif; ^nil
} !!

/* Invert the selected text, obtaining the display context.  */
Def  invertSelText(self)
{	dragDC := getContext(self);
	invSelTxt(self);
	releaseContext(self, dragDC);
} !!

/* Initialize the selection parameters according to xPos and yPos. */
 Def  initSelParms(self)
{  startChar := endChar := xPos;
  startLine := endLine := yPos + topLine;
}!!

/* Initialize the dragging parameters. */
Def beginDrag(self, wp, pt)
{ setFocus(self);
  hideCaret(self);
  Call SetCapture(hWnd);
  dragDC := getContext(self);
  invSelTxt(self);
  Call SelectObject(dragDC, stock(BLACK_BRUSH));
  setCurPos(self,pt);
  initSelParms(self);
  dragLine := startLine;
  pOrigin := point(x(self), y(self));
  oldX := pOrigin.x
}!!


/* Handle case where mouse is dragged 
  down one or more lines. */
Def  dragDown(self)
{
  if size(workText[dragLine]) > 0 or
    startLine >= yPos + topLine
  then invertLine(self, oldX, pOrigin.y,
    size(workText[dragLine]) * tmWidth + 2
    - oldX)
  else selNulLine(self, pOrigin.y)
  endif;
  do(over(1+dragLine-topLine, yPos),
  {using(line)     invertLine(self, 2,
    line*tmHeight+2,
    size(workText[line+topLine]) * tmWidth);
  } );
  dragLine := yPos + topLine;
  pOrigin.y := y(self);
  oldX := x(self);
  if size(workText[dragLine]) > 0
  then invertLine(self, 2, pOrigin.y,
    oldX - 2)
  else selNulLine(self, pOrigin.y)
  endif;
} !!

/* Handle case where mouse is dragged up one or more lines. */
Def dragUp(self)
{
  if size(workText[dragLine]) > 0 or
    startLine <= yPos + topLine
  then invertLine(self, 2, pOrigin.y, oldX - 2);
  else selNulLine(self, pOrigin.y);
  endif;
  do(over(yPos+1, dragLine-topLine),
  {using(line)
    invertLine(self, 2, line*tmHeight+2,   
    size(workText[line+topLine])  * tmWidth);
  } );
  dragLine := yPos + topLine;
  pOrigin.x := size(workText[dragLine]) * tmWidth + 2;
  pOrigin.y := y(self);
  oldX := x(self);
  if size(workText[dragLine]) > 0
  then invertLine(self, pOrigin.x,     
    pOrigin.y, oldX - pOrigin.x);
  else selNulLine(self, pOrigin.y);
  endif;
} !!


/* Show selected text while the mouse is dragged. */
Def drag(self, wp, pt | yRef)
{	setCurPos(self, pt);
	yRef := dragLine - topLine;
 select
	case (yPos == yRef) and (x(self) <> oldX) and 
	     	(xPos < size(workText[yPos+topLine]) + 1)
	is	invertLine(self, oldX, pOrigin.y, x(self) - oldX);
		oldX := x(self)
	endCase

	case yPos > yRef
	is   dragDown(self)
	endCase
	
	case yPos < yRef
	is   dragUp(self)
	endCase
 endSelect
}!!

/* Stop selecting text, move cursor. */
Def endDrag(self, wp, pt | curline)
{	Call ReleaseCapture();
	releaseContext(self, dragDC);
	setCurPos(self,pt);  curline := yPos + topLine;
 select 
	case  curline == startLine
	is startChar := min(startChar, xPos);
	   endChar := max(endChar, xPos);
	endCase
	case curline < startLine
	is startLine := curline;  startChar := xPos;
	endCase
	default  endLine := curline; endChar := xPos;
 endSelect; 
	moveCaret(self);
	showCaret(self)
}!!

/* Select all text in workText. */
Def selectAll(self)
{ startLine := startChar := 0;
  endChar := limit(last(workText));
  endLine := workText.lastElement - 1;
} !!

/* Pass any eol messages on to the Display window. */
Def eol(self)
{ eol(TheApp.display);
}!!

