/* Remote target communications for NetBSD targets via UDP.
   Copyright (C) 1988-1991 Free Software Foundation, Inc.
   Copyright (C) 1993, 1995 Wolfgang Solfrank.
   Copyright (C) 1993, 1995 TooLs GmbH.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foudation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that is will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Remote communication protocol.

   Data is send binary.
   The first byte is the command code.

   Packets have a sequence number and the data length prepended
   and a checksum byte appended.
   The answer has the same sequence number.  On a timeout, packets
   are rerequested by resending the request.

	Request		Packet

	read registers	R
	reply		pXX...X		Register data is sent in target format.
			emessage	for an error.

	write regs	WXX...X		Same as above.
	reply		ok		for success.
			emessage	for an error.
	
	read mem	MAA...ALL...L	AA...A is address, LL...L is length
					(both as CORE_ADDR).
	reply		pXX...X		XX...X is mem contents.
			emessage	for an error.
	
	write mem	NAA...ALL...LXX...X
					AA...A is address, LL...L is length,
					XX...X is data.
	reply		ok		for success.
			emessage	for an error.

	cont		C

	step		S

	open debug	O

	There is no immediate reply to step, cont or open.
	The reply comes when the machines stops (again).
	It is		s

	detach		X
	reply		ok
*/

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include "defs.h"
#include "frame.h"
#include "inferior.h"
#include "target.h"
#include "wait.h"
#include "terminal.h"

#ifdef	USG
#include <sys/types.h>
#endif

#include <signal.h>

extern void start_remote();

extern struct target_ops bsd_ops;	/* Forward decl */

static int timeout = 1;

/* Descriptor for I/O to remote machine.  Initialize it to -1 so that
   remote_open knows that we don't have a file open when the program
   starts.  */
static int bsd_desc = -1;

#define	PBUFSIZ	400

/* Maximum number of bytes to read/write at once.  */
#define	MAXBUFBYTES	(PBUFSIZ-10)

static void remote_send();
static void putpkt();
static void getpkt();
static int netopen();
static void netread();
static void netwrite();


/* Initialize remote connection */
void
bsd_start()
{
}

/* Clean up connection to remote debugger.  */
/* ARGSUSED */
void
bsd_close(quitting)
int quitting;
{
    if (bsd_desc >= 0)
	close(bsd_desc);
    bsd_desc = -1;
}

/* Open a connection to a remote debugger.
   NAME is the name of the target machines.  */
void
bsd_open(name,from_tty)
char *name;
int from_tty;
{
    if (name == 0)
	error(
"To open a remote debug connection, you need to specify\n\
the name of the target machines.");

    target_preopen(from_tty);

    bsd_close(0);

    if (netopen(name,from_tty) < 0)
	perror_with_name(name);
}

/* bsd_detach()
   takes a program previously attached to and detaches it.
   We better not have left any breakpoints
   in the program or it'll die when it hits one.
   Close the open connection to the remote debugger.
   Use this when you want to detach and do something else
   with your gdb.  */
static void
bsd_detach(args,from_tty)
char *args;
int from_tty;
{
    char buf[PBUFSIZ];
    int l;

    if (args)
	error("Argument given to \"detach\" when remotely debugging BSD.");
    
    *buf = 'X';
    l = 1;
    remote_send(buf,&l);

    pop_target();
    if (from_tty)
	puts_filtered("Ending remote debugging BSD.\n");
}

/* Tell the remote machine to resume.  */
void
bsd_resume(pid,step,siggnal)
int pid;
int step;
enum target_signal siggnal;
{
    if (siggnal != TARGET_SIGNAL_0)
	error("Can't send signals to a remote BSD system.");
    
    putpkt(step ? "S" : "C",1);
}

/* Wait until the remote machines stops, then return,
   storing status in STATUS just as `wait' would.
   Returns "pid" (though it's not clear what, if anything, that
   means in the case of this target).  */
int
bsd_wait(pid,status)
int pid;
struct target_waitstatus *status;
{
    unsigned char buf[PBUFSIZ];
    int l;

    status->kind = TARGET_WAITKIND_EXITED;
    status->value.integer = 0;
    getpkt(buf,&l);
    if (buf[0] == 'e') {
	buf[l] = 0;
	error("Remote failuer reply: %s",buf + 1);
    }
    if (buf[0] != 's')
	error("Invalid remote reply: %s",buf);
    status->kind = TARGET_WAITKIND_STOPPED;
    status->value.sig = TARGET_SIGNAL_TRAP;
    return 0;
}

/* Read the remote registers into the block REGS.  */
/* Currently we just read all the registers, so we don't use regno.  */
/* ARGSUSED */
void
bsd_fetch_registers(regno)
int regno;
{
    char buf[PBUFSIZ];
    int i;

    *buf = 'R';
    i = 1;
    remote_send(buf,&i);

    /* Reply describes registers byte by byte.
       Suck them all up, then supply them to the
       register cacheing/storage mechanism.  */
    for (i = 0; i < NUM_REGS; i++)
	supply_register(i,&buf[1 + REGISTER_BYTE(i)]);
}

/* Prepare to store registers.  Since we send them all, we have to
   read out the ones we don't want to change first.  */
void
bsd_prepare_to_store()
{
    bsd_fetch_registers(-1);
}

/* Store the remote registers from the contents of the block REGISTERS.  */
/* ARGSUSED */
void
bsd_store_registers(regno)
int regno;
{
    char buf[PBUFSIZ];
    int l;

    buf[0] = 'W';

    bcopy(registers,buf + 1,REGISTER_BYTES);
    l = 1 + REGISTER_BYTES;

    remote_send(buf,&l);
}

/* Put addr into buf */
static void
addrput(addr,buf)
CORE_ADDR addr;
char *buf;
{
    int i;
    
    buf += sizeof addr;
    for (i = sizeof addr; --i >= 0; addr >>= 8)
	*--buf = addr;
}

/* Write memory data directly to the remote machine.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes.  */
void
bsd_write_bytes(memaddr,myaddr,len)
CORE_ADDR memaddr;
char *myaddr;
int len;
{
    char buf[PBUFSIZ];

    if (len > PBUFSIZ - 1 - 2 * sizeof(CORE_ADDR))
	abort();
    
    *buf = 'N';
    addrput(memaddr,buf + 1);
    addrput((CORE_ADDR)len,buf + 1 + sizeof(CORE_ADDR));

    bcopy(myaddr,buf + 1 + 2 * sizeof(CORE_ADDR),len);

    len += 1 + 2 * sizeof(CORE_ADDR);
    remote_send(buf,&len);
}

/* Read memory data directly from the remote machine.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes.  */
void
bsd_read_bytes(memaddr,myaddr,len)
CORE_ADDR memaddr;
char *myaddr;
int len;
{
    char buf[PBUFSIZ];
    int i;
    char *p;

    if (len > PBUFSIZ - 1)
	abort();
    
    *buf = 'M';
    addrput(memaddr,buf + 1);
    addrput((CORE_ADDR)len,buf + 1 + sizeof(CORE_ADDR));

    i = 1 + 2 * sizeof(CORE_ADDR);
    remote_send(buf,&i);

    if (i != len + 1)
	error("bsd_read_bytes got wrong size of answer");
    
    /* Reply describes bytes in target format */
    bcopy(buf + 1,myaddr,len);
}

/* Read or write LEN bytes from inferior memory at MEMADDR, transferring
   to or from debugger address MYADDR.  Write to inferior if SHOULD_WRITE is
   nonzero.  Returns length of data written or read; 0 for error.  */
/* ARGSUSED */
int
bsd_xfer_memory(memaddr,myaddr,len,should_write,target)
CORE_ADDR memaddr;
char *myaddr;
int len;
int should_write;
struct target_ops *target;			/* ignored */
{
    int origlen = len;
    int xfersize;

    while (len > 0) {
	xfersize = min(len,MAXBUFBYTES);

	if (should_write)
	    bsd_write_bytes(memaddr,myaddr,xfersize);
	else
	    bsd_read_bytes(memaddr,myaddr,xfersize);
	memaddr	+= xfersize;
	myaddr	+= xfersize;
	len	-= xfersize;
    }
    return origlen;	/* no error possible */
}

void
bsd_files_info()
{
    printf("bsd files info missing here.  FIXME.\n");
}

static char bpt[] = BREAKPOINT;

int
bsd_insert_breakpoint(addr,save)
CORE_ADDR addr;
char *save;
{
    bsd_read_bytes(addr,save,sizeof bpt);
    bsd_write_bytes(addr,bpt,sizeof bpt);
    return 0;
}

int
bsd_remove_breakpoint(addr,save)
CORE_ADDR addr;
char *save;
{
    bsd_write_bytes(addr,save,sizeof bpt);
    return 0;
}

/* Send the command in BUF to the remote machine,
   and read the reply into BUF.
   Report an error if we get an error reply.  */
static void
remote_send(buf,lp)
char *buf;
int *lp;
{
    putpkt(buf,*lp);
    getpkt(buf,lp);

    if (buf[0] == 'e') {
	buf[*lp] = 0;
	error("Remote failure reply: %s",buf + 1);
    }
}

/* Send a packet to the remote machine, with error checking.
   The data of the packet is in BUF.  */
static void
putpkt(buf,l)
char *buf;
int l;
{
    int i;
    unsigned char csum = 0;
    char buf2[500];
    char *p;

    /* Copy the packet info buffer BUF2, encapsulating it
       and giving it a checksum.  */

    p = buf2;

    for (i = 0; i < l; i++) {
	csum += buf[i];
	*p++ = buf[i];
    }
    *p++ = csum;

    /* Send it over and over until we get a positive ack.  */
    netwrite(buf2,l);
}

/* Read a packet from the remote machine, with error checking,
   and store it in BUF.  */
static void
getpkt(buf,lp)
char *buf;
int *lp;
{
    netread(buf,lp);
}

/* net I/O routines */
/* open a network connection to the remote machine */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <setjmp.h>

static struct sockaddr_in sin;
static netseq;
static jmp_buf netenv;

static void
nettimer()
{
    longjmp(netenv,1);
}

static int
netopen(name,from_tty)
char *name;
int from_tty;
{
    struct hostent *he;
    char buf[512];
    char hostname[32];
    char *cp;
    int csum;

    if (!(he = gethostbyname(name)))
	return -1;

    if ((bsd_desc = socket(AF_INET,SOCK_DGRAM,0)) < 0)
	fatal("netopen: socket failed\n");
    
    sin.sin_family = AF_INET;
    sin.sin_port = 1138;
    sin.sin_addr.s_addr = 0;
    if (bind(bsd_desc,&sin,sizeof sin) < 0)
	fatal("netopen: bind failed\n");

    sin.sin_addr = *(struct in_addr *)he->h_addr;

    netseq = 0;

    if (from_tty)
	printf("Remote debugging on %s\n",name);
    push_target(&bsd_ops);	/* Switch to using remote target now.  */
    /* Set up read timeout timer.  */
    if ((void (*)())signal(SIGALRM,nettimer) == (void (*)())-1)
	perror("netopen: error in signal");

    /* Send initial open message */
    gethostname(hostname,sizeof hostname);
    sprintf(buf,"O%s:%s",cuserid(0),hostname);
    csum = 0;
    for (cp = buf; *cp; csum += *cp++);
    *cp = csum;
    netwrite(buf,cp - buf);

    start_remote();
    return bsd_desc;
}

static struct netbuf {
    long seq;
    short len;
    char buf[512];
} ibuf, obuf;
static int olen;

static void
netwrite(buf,len)
char *buf;
int len;
{
    struct sockaddr_in frm;
    int l;

    if (len > sizeof(obuf.buf))
	error("netwrite: packet too large\n");
    
    obuf.seq = htonl(netseq);
    obuf.len = htons(len);
    memcpy(obuf.buf,buf,len + 1);
    olen = len + sizeof(obuf.seq) + sizeof(obuf.len) + 1;
    if (sendto(bsd_desc,&obuf,olen,0,&sin,sizeof sin) < 0)
	fatal("netwrite: sendto\n");
}

static void
netread(buf,lp)
char *buf;
int *lp;
{
    int len, l, i, csum;
    char *cp;
    struct sockaddr_in frm;

    while (1) {
	if (setjmp(netenv)) {
	    /* assume we didn't get the packet */
	    if (sendto(bsd_desc,&obuf,olen,0,&sin,sizeof sin) < 0)
		fatal("netread: sendto\n");
	}
	alarm(timeout);
	l = sizeof frm;
	if ((len = recvfrom(bsd_desc,&ibuf,sizeof ibuf,0,&frm,&l)) < 0)
	    if (errno != EINTR)
		fatal("netread: recvfrom\n");
	if (ntohl(ibuf.seq) == netseq) {
	    alarm(0);
	    csum = 0;
	    cp = ibuf.buf;
	    len = ntohs(ibuf.len);
	    for (i = len; --i >= 0; csum += *cp++);
	    if ((char)csum != *cp)
		longjmp(netenv,1);
	    netseq++;
	    memcpy(buf,ibuf.buf,len);
	    *lp = len;
	    return;
	}
    }
}

static void
bsd_kill()
{
    target_mourn_inferior();
}

static void
bsd_mourn()
{
    unpush_target(&bsd_ops);
    generic_mourn_inferior();
}

/* Define the target subroutine names */
struct target_ops bsd_ops = {
    "bsd",
    "Remote BSD target via UDP/IP",
    "Debug a remote BSD computer via UDP/IP.\n\
Specify the name of the target machine",
    bsd_open,
    bsd_close,
    0,					/* attach */
    bsd_detach,
    bsd_resume,
    bsd_wait,
    bsd_fetch_registers,
    bsd_store_registers,
    bsd_prepare_to_store,
    bsd_xfer_memory,
    bsd_files_info,
    bsd_insert_breakpoint,
    bsd_remove_breakpoint,
    0,					/* terminal init */
    0,					/* terminal inferior */
    0,					/* terminal ours for output */
    0,					/* terminal ours */
    0,					/* terminal info */
    bsd_kill,
    0,					/* load */
    0,					/* lookup_symbol */
    0,					/* create_inferior */
    bsd_mourn,
    0,					/* can run */
    0,					/* notice signals */
    0,					/* thread alive */
    0,					/* stop */
    process_stratum,
    0,					/* next */
    1,					/* has all mem */
    1,					/* has mem */
    1,					/* has stack */
    1,					/* has regs */
    1,					/* has exec */
    0,					/* sections */
    0,					/* sections end */
    OPS_MAGIC				/* Always the last thing */
};

void
_initialize_bsd()
{
    add_target(&bsd_ops);
}
