/***********************************************************************
*
* asmoptab.c - Opcode table for the IBM 7090 assembler.
*
* Changes:
*   05/21/03   DGP   Original.
*   12/08/04   DGP   Started adding MAP pseudo ops.
*   02/02/05   DGP   Added some missing opcodes, THREE, SEVEN & ZERO.
*	
***********************************************************************/

#include <string.h>
#include <memory.h>
#include <stdlib.h>

#include "asmdef.h"

/*
** Standard opcodes
*/

OpCode optable[NUMOPS] =
{
   { "ABS",    ABS_T,    0,     TYPE_P    },
   { "ACL",    00361,    0,     TYPE_B    },
   { "ADD",    00400,    0,     TYPE_B    },
   { "ADM",    00401,    0,     TYPE_B    },
   { "ALS",    00767,    0,     TYPE_B    },
   { "ANA",    04320,    0,     TYPE_B    },
   { "ANS",    00320,    0,     TYPE_B    },
   { "ARS",    00771,    0,     TYPE_B    },
   { "AXC",    04774,    0,     TYPE_B    },
   { "AXT",    00774,    0,     TYPE_B    },
   { "BCD",    BCD_T,    0,     TYPE_P    },
   { "BCI",    BCI_T,    0,     TYPE_P    },
   { "BEGIN",  BEGIN_T,  0,     TYPE_P    },
   { "BES",    BES_T,    0,     TYPE_P    },
   { "BFT",    BFT_T,    0,     TYPE_P    },
   { "BNT",    BNT_T,    0,     TYPE_P    },
   { "BOOL",   BOOL_T,   0,     TYPE_P    },
   { "BRA",    07000,    0,     TYPE_A    },
   { "BRN",    03000,    0,     TYPE_A    },
   { "BSF",    04764,    0,     TYPE_E    },
   { "BSR",    00764,    0,     TYPE_E    },
   { "BSS",    BSS_T,    0,     TYPE_P    },
   { "BTT",    00760,    0,     TYPE_E    },
   { "CAD",    04700,    0,     TYPE_B    },
   { "CAL",    04500,    0,     TYPE_B    },
   { "CALL",   CALL_T,   0,     TYPE_P    },
   { "CAQ",    04114,    0,     TYPE_C    },
   { "CAS",    00340,    0,     TYPE_B    },
   { "CFF",    00760,    00030, TYPE_E    },
   { "CHS",    00760,    00002, TYPE_E    },
   { "CLA",    00500,    0,     TYPE_B    },
   { "CLM",    00760,    00000, TYPE_E    },
   { "CLS",    00502,    0,     TYPE_B    },
   { "COM",    00760,    00006, TYPE_E    },
   { "COMMON", COMMON_T, 0,     TYPE_P    },
   { "CONTRL", CONTRL_T, 0,     TYPE_P    },
   { "COUNT",  COUNT_T,  0,     TYPE_P    },
   { "CPY",    00700,    0,     TYPE_B    },
   { "CPYD",   05000,    0,     TYPE_CHAN },
   { "CPYP",   04000,    0,     TYPE_CHAN },
   { "CRQ",    04154,    0,     TYPE_C    },
   { "CTL",    02000,    0,     TYPE_CHAN },
   { "CTLR",   02000,    02,    TYPE_CHAN },
   { "CTLW",   02400,    00,    TYPE_CHAN },
   { "CVR",    00114,    0,     TYPE_C    },
   { "DCT",    00760,    00012, TYPE_E    },
   { "DEBM",   01210,    01212, TYPE_DISK },
   { "DEC",    DEC_T,    0,     TYPE_P    },
   { "DETAIL", DETAIL_T, 0,     TYPE_P    },
   { "DFAD",   00301,    0,     TYPE_B    },
   { "DFAM",   00305,    0,     TYPE_B    },
   { "DFDH",   04240,    0,     TYPE_B    },
   { "DFDP",   04241,    0,     TYPE_B    },
   { "DFMP",   00261,    0,     TYPE_B    },
   { "DFSB",   00303,    0,     TYPE_B    },
   { "DFSM",   00307,    0,     TYPE_B    },
   { "DLD",    00443,    0,     TYPE_B    },
   { "DNOP",   01212,    01212, TYPE_DISK },
   { "DREL",   01204,    01212, TYPE_DISK },
   { "DSAI",   01007,    01212, TYPE_DISK },
   { "DSBM",   01211,    01212, TYPE_DISK },
   { "DSEK",   01012,    01212, TYPE_DISK },
   { "DST",    04603,    0,     TYPE_B    },
   { "DUAM",   04305,    0,     TYPE_B    },
   { "DUFA",   04301,    0,     TYPE_B    },
   { "DUFM",   04261,    0,     TYPE_B    },
   { "DUFS",   04303,    0,     TYPE_B    },
   { "DUP",    DUP_T,    0,     TYPE_P    },
   { "DUSM",   04307,    0,     TYPE_B    },
   { "DVCY",   01005,    01212, TYPE_DISK },
   { "DVH",    00220,    0,     TYPE_B    },
   { "DVHA",   01011,    01212, TYPE_DISK },
   { "DVP",    00221,    0,     TYPE_B    },
   { "DVSR",   01002,    01212, TYPE_DISK },
   { "DVTA",   01010,    01212, TYPE_DISK },
   { "DVTN",   01004,    01212, TYPE_DISK },
   { "DWRC",   01006,    01212, TYPE_DISK },
   { "DWRF",   01003,    01212, TYPE_DISK },
   { "ECTM",   04760,    00006, TYPE_E    },
   { "EFTM",   04760,    00002, TYPE_E    },
   { "EJECT",  EJE_T,    0,     TYPE_P    },
   { "EMTM",   04760,    00016, TYPE_E    },
   { "ENB",    00564,    0,     TYPE_B    },
   { "END",    END_T,    0,     TYPE_P    },
   { "ENDM",   ENDM_T,   0,     TYPE_P    },
   { "ENDQ",   ENDQ_T,   0,     TYPE_P    },
   { "ENK",    00760,    00004, TYPE_E    },
   { "ENTRY",  ENT_T,    0,     TYPE_P    },
   { "EQU",    EQU_T,    0,     TYPE_P    },
   { "ERA",    00322,    0,     TYPE_B    },
   { "ESNT",   04021,    0,     TYPE_B    },
   { "ESTM",   04760,    00005, TYPE_E    },
   { "ETC",    ETC_T,    0,     TYPE_P    },
   { "ETM",    00760,    00007, TYPE_E    },
   { "ETT",    04760,    0,     TYPE_E    },
   { "EVEN",   EVEN_T,   0,     TYPE_P    },
   { "EXTERN", EXT_T,    0,     TYPE_P    },
   { "FAD",    00300,    0,     TYPE_B    },
   { "FAM",    00304,    0,     TYPE_B    },
   { "FDH",    00240,    0,     TYPE_B    },
   { "FDP",    00241,    0,     TYPE_B    },
   { "FILE",   FILE_T,   0,     TYPE_P    },
   { "FIVE",   05000,    0,     TYPE_A    },
   { "FMP",    00260,    0,     TYPE_B    },
   { "FOR",    04000,    0,     TYPE_A    },
   { "FOUR",   04000,    0,     TYPE_A    },
   { "FRN",    00760,    00011, TYPE_E    },
   { "FSB",    00302,    0,     TYPE_B    },
   { "FSM",    00306,    0,     TYPE_B    },
   { "FUL",    FUL_T,    0,     TYPE_P    },
   { "FVE",    05000,    0,     TYPE_A    },
   { "GOTO",   GOTO_T,   0,     TYPE_P    },
   { "HEAD",   HEAD_T,   0,     TYPE_P    },
   { "HPR",    00420,    0,     TYPE_B    },
   { "HTR",    00000,    0,     TYPE_B    },
   { "ICC",    07000,    02,    TYPE_CHAN },
   { "IFF",    IFF_T,    0,     TYPE_P    },
   { "IFT",    IFT_T,    0,     TYPE_P    },
   { "IIA",    00041,    0,     TYPE_D    },
   { "IIB",    IIB_T,    0,     TYPE_P    },
   { "IIL",    04051,    0,     TYPE_D    },
   { "IIR",    00051,    0,     TYPE_D    },
   { "IIS",    00440,    0,     TYPE_B    },
   { "INDEX",  INDEX_T,  0,     TYPE_P    },
   { "IOCD",   00000,    0,     TYPE_A    },
   { "IOCDN",  00000,    02,    TYPE_A    },
   { "IOCP",   04000,    0,     TYPE_A    },
   { "IOCPN",  04000,    02,    TYPE_A    },
   { "IOCT",   05000,    0,     TYPE_A    },
   { "IOCTN",  05000,    02,    TYPE_A    },
   { "IORP",   02000,    0,     TYPE_A    },
   { "IORPN",  02000,    02,    TYPE_A    },
   { "IORT",   03000,    0,     TYPE_A    },
   { "IORTN",  03000,    02,    TYPE_A    },
   { "IOSP",   06000,    0,     TYPE_A    },
   { "IOSPN",  06000,    02,    TYPE_A    },
   { "IOST",   07000,    0,     TYPE_A    },
   { "IOSTN",  07000,    02,    TYPE_A    },
   { "IOT",    00760,    00005, TYPE_E    },
   { "IRP",    IRP_T,    0,     TYPE_P    },
   { "KEEP",   KEEP_T,   0,     TYPE_P    },
   { "LABEL",  LABEL_T,  0,     TYPE_P    },
   { "LAC",    00535,    0,     TYPE_B    },
   { "LAR",    03000,    0,     TYPE_CHAN },
   { "LAS",    04340,    0,     TYPE_B    },
   { "LBL",    LBL_T,    0,     TYPE_P    },
   { "LBOOL",  LBOOL_T,  0,     TYPE_P    },
   { "LBT",    00760,    00001, TYPE_E    },
   { "LCC",    06400,    02,    TYPE_CHAN },
   { "LCHA",   00544,    0,     TYPE_B    },
   { "LCHB",   04544,    0,     TYPE_B    },
   { "LCHC",   00545,    0,     TYPE_B    },
   { "LCHD",   04545,    0,     TYPE_B    },
   { "LCHE",   00546,    0,     TYPE_B    },
   { "LCHF",   04546,    0,     TYPE_B    },
   { "LCHG",   00547,    0,     TYPE_B    },
   { "LCHH",   04547,    0,     TYPE_B    },
   { "LDA",    00460,    0,     TYPE_B    },
   { "LDC",    04535,    0,     TYPE_B    },
   { "LDI",    00441,    0,     TYPE_B    },
   { "LDIR",   LDIR_T,   0,     TYPE_P    },
   { "LDQ",    00560,    0,     TYPE_B    },
   { "LFT",    04054,    0,     TYPE_D    },
   { "LFTM",   04760,    00004, TYPE_E    },
   { "LGL",    04763,    0,     TYPE_B    },
   { "LGR",    04765,    0,     TYPE_B    },
   { "LIP",    06000,    02,    TYPE_CHAN },
   { "LIPT",   01000,    02,    TYPE_CHAN },
   { "LIST",   LIST_T,   0,     TYPE_P    },
   { "LIT",    LIT_T,    0,     TYPE_P    },
   { "LITORG", LITORG_T, 0,     TYPE_P    },
   { "LLS",    00763,    0,     TYPE_B    },
   { "LMTM",   00760,    00016, TYPE_E    },
   { "LNT",    04056,    0,     TYPE_D    },
   { "LOC",    LOC_T,    0,     TYPE_P    },
   { "LORG",   LORG_T,   0,     TYPE_P    },
   { "LRS",    00765,    0,     TYPE_B    },
   { "LSNM",   04760,    00010, TYPE_E    },
   { "LTM",    04760,    00007, TYPE_E    },
   { "LXA",    00534,    0,     TYPE_B    },
   { "LXD",    04534,    0,     TYPE_B    },
   { "MACRO",  MACRO_T,  0,     TYPE_P    },
   { "MAX",    MAX_T,    0,     TYPE_P    },
   { "MIN",    MIN_T,    0,     TYPE_P    },
   { "MON",    05000,    0,     TYPE_A    },
   { "MPR",    04200,    0,     TYPE_B    },
   { "MPY",    00200,    0,     TYPE_B    },
   { "MSE",    00760,    00000, TYPE_E    },
   { "MTH",    07000,    0,     TYPE_A    },
   { "MTW",    06000,    0,     TYPE_A    },
   { "MZE",    04000,    0,     TYPE_A    },
   { "NOCRS",  NOCRS_T,  0,     TYPE_P    },
   { "NOP",    00761,    0,     TYPE_B    },
   { "NTR",    01000,    0,     TYPE_A    },
   { "NULL",   NULL_T,   0,     TYPE_P    },
   { "NZT",    04520,    0,     TYPE_B    },
   { "OAI",    00043,    0,     TYPE_B    },
   { "OCT",    OCT_T,    0,     TYPE_P    },
   { "OFT",    00444,    0,     TYPE_B    },
   { "ONE",    01000,    0,     TYPE_A    },
   { "ONT",    00446,    0,     TYPE_B    },
   { "OPD",    OPD_T,    0,     TYPE_P    },
   { "OPSYN",  OPSYN_T,  0,     TYPE_P    },
   { "OPVFD",  OPVFD_T,  0,     TYPE_P    },
   { "ORA",    04501,    0,     TYPE_B    },
   { "ORG",    ORG_T,    0,     TYPE_P    },
   { "ORGCRS", ORGCRS_T, 0,     TYPE_P    },
   { "ORS",    04602,    0,     TYPE_B    },
   { "OSI",    00442,    0,     TYPE_B    },
   { "PAC",    00737,    0,     TYPE_B    },
   { "PAI",    00044,    0,     TYPE_B    },
   { "PAX",    00734,    0,     TYPE_B    },
   { "PBT",    04760,    00001, TYPE_E    },
   { "PCA",    00756,    0,     TYPE_B    }, 
   { "PCC",    PCC_T,    0,     TYPE_P    },
   { "PCD",    04756,    0,     TYPE_B    },
   { "PCG",    PCG_T,    0,     TYPE_P    },
   { "PDC",    04737,    0,     TYPE_B    },
   { "PDX",    04734,    0,     TYPE_B    },
   { "PIA",    04046,    0,     TYPE_B    },
   { "PMC",    PMC_T,    0,     TYPE_P    },
   { "PON",    01000,    0,     TYPE_A    },
   { "PSE",    00760,    00000, TYPE_E    },
   { "PTH",    03000,    0,     TYPE_A    },
   { "PTW",    02000,    0,     TYPE_A    },
   { "PUNCH",  PUNCH_T,  0,     TYPE_P    },
   { "PURGE",  PURGE_T,  0,     TYPE_P    },
   { "PXA",    00754,    0,     TYPE_B    },
   { "PXD",    04754,    0,     TYPE_B    },
   { "PZE",    00000,    0,     TYPE_A    },
   { "QUAL",   QUAL_T,   0,     TYPE_P    },
   { "RBOOL",  RBOOL_T,  0,     TYPE_P    },
   { "RCD",    00762,    00321, TYPE_E    },
   { "RCHA",   00540,    0,     TYPE_B    },
   { "RCHB",   04540,    0,     TYPE_B    },
   { "RCHC",   00541,    0,     TYPE_B    },
   { "RCHD",   04541,    0,     TYPE_B    },
   { "RCHE",   00542,    0,     TYPE_B    },
   { "RCHF",   04542,    0,     TYPE_B    },
   { "RCHG",   00543,    0,     TYPE_B    },
   { "RCHH",   04543,    0,     TYPE_B    },
   { "RCT",    00760,    00014, TYPE_E    },
   { "RDC",    00760,    00352, TYPE_E    },
   { "RDR",    00762,    00300, TYPE_E    },
   { "RDS",    00762,    0,     TYPE_E    },
   { "REF",    REF_T,    0,     TYPE_P    },
   { "REM",    REM_T,    0,     TYPE_P    },
   { "RETURN", RETURN_T, 0,     TYPE_P    },
   { "REW",    00772,    0,     TYPE_B    },
   { "RFT",    00054,    0,     TYPE_D    },
   { "RIA",    04042,    0,     TYPE_D    },
   { "RIB",    RIB_T,    0,     TYPE_P    },
   { "RIC",    00760,    00350, TYPE_E    },
   { "RIL",    04057,    0,     TYPE_D    },
   { "RIR",    00057,    0,     TYPE_D    },
   { "RIS",    00445,    0,     TYPE_B    },
   { "RND",    00760,    00010, TYPE_E    },
   { "RNT",    00056,    0,     TYPE_D    },
   { "RPR",    00762,    00361, TYPE_E    },
   { "RQL",    04773,    0,     TYPE_B    },
   { "RSCA",   00540,    0,     TYPE_B    },
   { "RSCB",   04540,    0,     TYPE_B    },
   { "RSCC",   00541,    0,     TYPE_B    },
   { "RSCD",   04541,    0,     TYPE_B    },
   { "RSCE",   00542,    0,     TYPE_B    },
   { "RSCF",   04542,    0,     TYPE_B    },
   { "RSCG",   00543,    0,     TYPE_B    },
   { "RSCH",   04543,    0,     TYPE_B    },
   { "RTB",    00762,    00220, TYPE_E    },
   { "RTD",    00762,    00200, TYPE_E    },
   { "RUN",    04772,    0,     TYPE_E    },
   { "SAR",    03000,    02,    TYPE_CHAN },
   { "SAVE",   SAVE_T,   0,     TYPE_P    },
   { "SAVEN",  SAVEN_T,  0,     TYPE_P    },
   { "SBM",    04400,    0,     TYPE_B    },
   { "SCA",    00636,    0,     TYPE_B    },
   { "SCD",    04636,    0,     TYPE_B    },
   { "SCHA",   00640,    0,     TYPE_B    },
   { "SCHB",   04640,    0,     TYPE_B    },
   { "SCHC",   00641,    0,     TYPE_B    },
   { "SCHD",   04641,    0,     TYPE_B    },
   { "SCHE",   00642,    0,     TYPE_B    },
   { "SCHF",   04642,    0,     TYPE_B    },
   { "SCHG",   00643,    0,     TYPE_B    },
   { "SCHH",   04643,    0,     TYPE_B    },
   { "SDN",    00776,    0,     TYPE_B    },
   { "SET",    SET_T,    0,     TYPE_P    },
   { "SEVEN",  07000,    0,     TYPE_A    },
   { "SIB",    SIB_T,    0,     TYPE_P    },
   { "SIL",    04055,    0,     TYPE_D    },
   { "SIR",    00055,    0,     TYPE_D    },
   { "SIX",    06000,    0,     TYPE_A    },
   { "SLF",    00760,    00140, TYPE_E    },
   { "SLN",    00760,    00140, TYPE_E    },
   { "SLQ",    04620,    0,     TYPE_B    },
   { "SLT",    04760,    00140, TYPE_E    },
   { "SLW",    00602,    0,     TYPE_B    },
   { "SMS",    07000,    0,     TYPE_CHAN },
   { "SNS",    02400,    02,    TYPE_CHAN },
   { "SPACE",  SPC_T,    0,     TYPE_P    },
   { "SPR",    00760,    00360, TYPE_E    },
   { "SPT",    00760,    00360, TYPE_E    },
   { "SPU",    00760,    00340, TYPE_E    },
   { "SSM",    04760,    00003, TYPE_E    },
   { "SSP",    00760,    00003, TYPE_E    },
   { "SST",    SST_T,    0,     TYPE_P    },
   { "STA",    00621,    0,     TYPE_B    },
   { "STCA",   00544,    0,     TYPE_E    },
   { "STCB",   04544,    0,     TYPE_E    },
   { "STCC",   00545,    0,     TYPE_E    },
   { "STCD",   04545,    0,     TYPE_E    },
   { "STCE",   00546,    0,     TYPE_E    },
   { "STCF",   04546,    0,     TYPE_E    },
   { "STCG",   00547,    0,     TYPE_E    },
   { "STCH",   04547,    0,     TYPE_E    },
   { "STD",    00622,    0,     TYPE_B    },
   { "STI",    00604,    0,     TYPE_B    },
   { "STL",    04625,    0,     TYPE_B    },
   { "STO",    00601,    0,     TYPE_B    },
   { "STP",    00630,    0,     TYPE_B    },
   { "STQ",    04600,    0,     TYPE_B    },
   { "STR",    05000,    0,     TYPE_A    },
   { "STT",    00625,    0,     TYPE_B    },
   { "STZ",    00600,    0,     TYPE_B    },
   { "SUB",    00402,    0,     TYPE_B    },
   { "SVN",    07000,    0,     TYPE_A    },
   { "SWT",    00760,    00160, TYPE_E    },
   { "SXA",    00634,    0,     TYPE_B    },
   { "SXD",    04634,    0,     TYPE_B    },
   { "SYN",    EQU_T,    0,     TYPE_P    },
   { "TCD",    TCD_T,    0,     TYPE_P    },
   { "TCH",    01000,    0,     TYPE_A    },
   { "TCM",    05000,    02,    TYPE_CHAN },
   { "TCNA",   04060,    0,     TYPE_B    },
   { "TCNB",   04061,    0,     TYPE_B    },
   { "TCNC",   04062,    0,     TYPE_B    },
   { "TCND",   04063,    0,     TYPE_B    },
   { "TCNE",   04064,    0,     TYPE_B    },
   { "TCNF",   04065,    0,     TYPE_B    },
   { "TCNG",   04066,    0,     TYPE_B    },
   { "TCNH",   04067,    0,     TYPE_B    },
   { "TCOA",   00060,    0,     TYPE_B    },
   { "TCOB",   00061,    0,     TYPE_B    },
   { "TCOC",   00062,    0,     TYPE_B    },
   { "TCOD",   00063,    0,     TYPE_B    },
   { "TCOE",   00064,    0,     TYPE_B    },
   { "TCOF",   00065,    0,     TYPE_B    },
   { "TCOG",   00066,    0,     TYPE_B    },
   { "TCOH",   00067,    0,     TYPE_B    },
   { "TDC",    06400,    0,     TYPE_CHAN },
   { "TEFA",   00030,    0,     TYPE_B    },
   { "TEFB",   04030,    0,     TYPE_B    },
   { "TEFC",   00031,    0,     TYPE_B    },
   { "TEFD",   04031,    0,     TYPE_B    },
   { "TEFE",   00032,    0,     TYPE_B    },
   { "TEFF",   04032,    0,     TYPE_B    },
   { "TEFG",   00033,    0,     TYPE_B    },
   { "TEFH",   04033,    0,     TYPE_B    },
   { "THREE",  03000,    0,     TYPE_A    },
   { "TIF",    00046,    0,     TYPE_B    },
   { "TIO",    00042,    0,     TYPE_B    },
   { "TITLE",  TITLE_T,  0,     TYPE_P    },
   { "TIX",    02000,    0,     TYPE_A    },
   { "TLQ",    00040,    0,     TYPE_B    },
   { "TMI",    04120,    0,     TYPE_B    },
   { "TNO",    04140,    0,     TYPE_B    },
   { "TNX",    06000,    0,     TYPE_B    },
   { "TNZ",    04100,    0,     TYPE_B    },
   { "TOV",    00140,    0,     TYPE_B    },
   { "TPL",    00120,    0,     TYPE_B    },
   { "TQO",    00161,    0,     TYPE_B    },
   { "TQP",    00162,    0,     TYPE_B    },
   { "TRA",    00020,    0,     TYPE_B    },
   { "TRCA",   00022,    0,     TYPE_B    },
   { "TRCB",   04022,    0,     TYPE_B    },
   { "TRCC",   00024,    0,     TYPE_B    },
   { "TRCD",   04024,    0,     TYPE_B    },
   { "TRCE",   00026,    0,     TYPE_B    },
   { "TRCF",   04026,    0,     TYPE_B    },
   { "TRCG",   00027,    0,     TYPE_B    },
   { "TRCH",   04027,    0,     TYPE_B    },
   { "TSX",    00074,    0,     TYPE_B    },
   { "TTL",    TTL_T,    0,     TYPE_P    },
   { "TTR",    00021,    0,     TYPE_B    },
   { "TWO",    02000,    0,     TYPE_A    },
   { "TWT",    03400,    0,     TYPE_CHAN },
   { "TXH",    03000,    0,     TYPE_A    },
   { "TXI",    01000,    0,     TYPE_A    },
   { "TXL",    07000,    0,     TYPE_A    },
   { "TZE",    00100,    0,     TYPE_B    },
   { "UAM",    04304,    0,     TYPE_B    },
   { "UFA",    04300,    0,     TYPE_B    },
   { "UFM",    04260,    0,     TYPE_B    },
   { "UFS",    04302,    0,     TYPE_B    },
   { "UNLIST", UNLIST_T, 0,     TYPE_P    },
   { "UNPNCH", UNPNCH_T, 0,     TYPE_P    },
   { "USE",    USE_T,    0,     TYPE_P    },
   { "USM",    04306,    0,     TYPE_B    },
   { "VDH",    00224,    0,     TYPE_C    },
   { "VDP",    00225,    0,     TYPE_C    },
   { "VFD",    VFD_T,    0,     TYPE_P    },
   { "VLM",    00204,    0,     TYPE_C    },
   { "WDR",    00766,    00300, TYPE_E    },
   { "WEF",    00770,    0,     TYPE_E    },
   { "WPB",    00766,    00362, TYPE_E    },
   { "WPD",    00766,    00361, TYPE_E    },
   { "WPR",    00766,    00361, TYPE_E    },
   { "WPU",    00766,    00341, TYPE_E    },
   { "WRS",    00766,    0,     TYPE_E    },
   { "WTB",    00766,    00220, TYPE_E    },
   { "WTD",    00766,    00200, TYPE_E    },
   { "WTR",    00000,    0,     TYPE_CHAN },
   { "WTV",    00766,    00030, TYPE_E    },
   { "XCA",    00131,    0,     TYPE_B    },
   { "XCL",    04130,    0,     TYPE_B    },
   { "XEC",    00522,    0,     TYPE_B    },
   { "XIT",    00021,    0,     TYPE_B    },
   { "XMT",    00000,    02,    TYPE_A    },
   { "ZAC",    04754,    0,     TYPE_B    },
   { "ZERO",   00000,    0,     TYPE_A    },
   { "ZET",    00520,    0,     TYPE_B    },
   { "ZSA",    00634,    0,     TYPE_B    },
   { "ZSD",    04634,    0,     TYPE_B    },
};

/*
** Added opcodes
*/

extern int opdefcount;		/* Number of user defined opcode in opdef */
extern OpDefCode *freeops;	/* Reusable opdef codes */
extern OpDefCode *opdefcode[MAXDEFOPS];/* The user defined opcode table */

static char retopcode[MAXSYMLEN+2]; /* Return opcode symbol buffer */
static OpCode retop;		/* Return opcode buffer */

/***********************************************************************
* freeopd - Link a opcode entry on free chain.
***********************************************************************/

void
freeopd (OpDefCode *or)
{
   or->next = freeops;
   freeops = or;
}

/***********************************************************************
* stdlookup - Lookup standard opcode. Uses binary search.
***********************************************************************/

static OpCode *
stdlookup (char *op)
{
   OpCode *ret = NULL;
   int done = FALSE;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

#ifdef DEBUGOP
   printf ("stdlookup: Entered: op = %s\n", op);
#endif

   lo = 0;
   up = NUMOPS;

   done = FALSE;
   while (!done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGOP
      printf (" mid = %d, last = %d\n", mid, last);
#endif
      if (last == mid) break;
      r = strcmp (optable[mid].opcode, op);
      if (r == 0)
      {
	 ret = &optable[mid];
	 done = TRUE;
      }
      else if (r < 0)
      {
	 lo = mid;
      }
      else 
      {
	 up = mid;
      }
      last = mid;
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* addlookup - Lookup added opcode. Uses binary search.
***********************************************************************/

static OpCode *
addlookup (char *op)
{
   OpCode *ret = NULL;
   int done = FALSE;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

#ifdef DEBUGOP
   printf ("addlookup: Entered: op = %s\n", op);
#endif

   lo = 0;
   up = opdefcount;
   
   while (opdefcount && !done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGOP
      printf (" mid = %d, last = %d\n", mid, last);
#endif
      if (opdefcount == 1) done = TRUE;
      else if (last == mid) break;
         
      r = strcmp (opdefcode[mid]->opcode, op);
      if (r == 0)
      {
	 strcpy (retopcode, op);
	 retop.opcode = retopcode;
	 retop.opvalue = opdefcode[mid]->opvalue;
	 retop.opmod = opdefcode[mid]->opmod;
	 retop.optype = opdefcode[mid]->optype;
	 ret = &retop;
	 done = TRUE;
      }
      else if (r < 0)
      {
	 lo = mid;
      }
      else 
      {
	 up = mid;
      }
      last = mid;
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* oplookup - Lookup opcode.
***********************************************************************/

OpCode *
oplookup (char *op)
{
   OpCode *ret = NULL;
   int i;
   int opch;
   char opbf[MAXSYMLEN+2];

#ifdef DEBUGOP
   printf ("oplookup: Entered: op = %s\n", op);
#endif

   /*
   ** Check added opcodes first, incase of override
   */

   if (!(ret = addlookup (op)))
   {
      /*
      ** Check standard opcode table
      */

      if (!(ret = stdlookup (op)))
      {

	 /*
	 ** Check if it's an I/O opcode. Like WPR[A-H]
	 */

	 strcpy (opbf, op);
	 i = strlen(opbf) - 1;
	 opch = opbf[i];
	 opbf[i] = '\0';
	 if (i == 3 && opch >= 'A' && opch <= 'H')
	 {
	    if ((ret = stdlookup (opbf)) != NULL)
	    {
	       strcpy (retopcode, op);
	       retop.opcode = retopcode;
	       retop.opvalue = ret->opvalue;
	       retop.opmod = ret->opmod | (opch - 'A' + 1) << 9;
	       retop.optype = ret->optype;
	       ret = &retop;
	    }
         }
      }
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* opadd - Add opcode.
***********************************************************************/

void
opadd (char *op, unsigned c0, unsigned c1, int type)
{
   OpDefCode *new;
   int lo, up;

#ifdef DEBUGOP
   printf ("opadd: Entered: op = %s, c0 = %4.4o, c1 = %8.8o, type = %d\n",
	    op, c0, c1, type);
#endif

   /*
   ** Allocate storage for opcode and fill it in
   */

   if (opdefcount+1 > MAXDEFOPS)
   {
      fprintf (stderr, "asm7090: Op Code table exceeded\n");
      exit (ABORT);
   }

   if (freeops)
   {
      new = freeops;
      freeops = new->next;
   }
   else if ((new = (OpDefCode *)malloc (sizeof (OpDefCode))) == NULL)
   {
      fprintf (stderr, "asm7090: Unable to allocate memory\n");
      exit (ABORT);
   }

   memset (new, '\0', sizeof (OpDefCode));
   strcpy (new->opcode, op);
   new->opvalue = c0;
   new->opmod = c1;
   new->optype = type;

   if (opdefcount == 0)
   {
      opdefcode[0] = new;
      opdefcount++;
      return;
   }

   /*
   ** Insert pointer in sort order.
   */

   for (lo = 0; lo < opdefcount; lo++)
   {
      if (strcmp (opdefcode[lo]->opcode, op) > 0)
      {
	 for (up = opdefcount + 1; up > lo; up--)
	 {
	    opdefcode[up] = opdefcode[up-1];
	 }
	 opdefcode[lo] = new;
	 opdefcount++;
	 return;
      }
   }
   opdefcode[opdefcount] = new;
   opdefcount++;
}

/***********************************************************************
* opdel - Delete an op code from the table.
***********************************************************************/

void
opdel (char *op)
{
   int i;

#ifdef DEBUGOP
   printf ("opdel: Entered: op = %s\n",
	    op);
#endif
   for (i = 0; i < opdefcount; i++)
   {
      if (!strcmp (opdefcode[i]->opcode, op))
      {
         free (opdefcode[i]);
	 for (; i < opdefcount; i++)
	 {
	    opdefcode[i] = opdefcode[i+1];
	 }
	 opdefcount --;
         return;
      }
   }
}
