/*----------------------------------------------------------------------------
/ ldio.c - a part of 'dsetup'
/        - perform i/o to a logical disk routines
/
/ 01-Apr-88  Craig J. Kim
/ LogicalDisk didn't point at sufficient memory location - fixed
/ Added comments
/---------------------------------------------------------------------------*/

#include "disksect.h"
#include "dsetup.h"
#include "dk.h"

#ifdef RO_UNIX
#include <sys/stat.h>
#define PERROR(x) perror(x)
#else
#define PERROR(x)
#endif

#ifndef REG
#define REG register
#endif


char ldbuf[BLKSIZE];

/*---------------------------------------------------- ldopen() --------------
/ opens a logical disk - if not standalone, just return error
/ check is done to such that there will be upto 4 controllers and 14 disks
/ per controller.  dev->dev_num stores a disk address in 00110011 format
/ where upper nibble is for which controller and lower for disk.
/---------------------------------------------------------------------------*/
ldopen(dev, ld)
struct icbdev *dev;
{
#ifdef STANDALONE
    REG int pdnum;                      /* 00110011 */
    REG char *p;
    REG char *LogicalDisk;
#define LV_CONTROLLER   1
#define LV_DISK         3
#define LV_SLICE        5

    LogicalDisk = ld == -1 ? "c0d0r" : "c0d0s00";
    pdnum = dev->dev_num;
    if (pdnum < 0 || pdnum > 0x3d)      /* between 0 and 55 (00111110) */
        error("disk number out of range");

    LogicalDisk[LV_CONTROLLER] = (pdnum >> 4) + '0';
    LogicalDisk[LV_DISK] = (pdnum & 3) + '0';
    p = &LogicalDisk[LV_SLICE];         /* which logical disk */
    if (ld > 9)                         /* this indicates that there may */
        *p++ = (ld / 10) + '0';         /*  be upto 99 slices */
    if (ld >= 0)
        *p++ = (ld % 10) + '0';
    *p = 0;

    xprintf("open(\"%s\", 2)\n", LogicalDisk);

    return(open(LogicalDisk, 2));       /* now attempt to open it */
#else
    return(-1);                         /* don't allow any direct i/o */
#endif
}

/*---------------------------------------------------- ldreadck() ------------
/ perform read on the specified record where a record is BLKSIZE in bytes
/ no data transfer to user buffer is performed.
/---------------------------------------------------------------------------*/
ldreadck(fd, s)
{
    lseek(fd, s * BLKSIZE, 0);          /* position the head */
    if (read(fd, ldbuf, BLKSIZE) < 0) { /* can we read a block? */
        PERROR("dsetup ");              /* nope! */
        errxit("ld read error");
    }
}

/*---------------------------------------------------- ldwriteck() -----------
/ perform write on the specified record where a record is BLKSIZE in bytes
/ no data transfer from user buffer is performed.
/---------------------------------------------------------------------------*/
ldwriteck(fd, s)
{
    lseek(fd, s * BLKSIZE, 0);          /* move head to specified location */
    if (write(fd, ldbuf, BLKSIZE) < 0) {/* can we write a block? */
        PERROR("dsetup ");              /* oops! */
        errxit("ld write error");
    }
}

/*---------------------------------------------------- ldread() --------------
/ perform read on given 'chs (cylinder/head/sector)'
/---------------------------------------------------------------------------*/
ldread(setup, chs)
REG struct setup *setup;
{
    int fd, ld, bno;

    bno = chs_to_ldb(setup, chs);       /* convert to block address */
    ld = (bno >> 24) & 0x1f;            /* which logical disk */
    if (ld == LOGDR + 1)                /* reserved area? */
        ld = -1;                        /* yes */

    xprintf("ldread($%x, %d:$%x)\n", chs, ld, bno);

    if (bno < 0 || ldb_to_chs(setup, ld, bno &= 0xffffff) != chs) {
        warn("$%x is skipped in Ldisk %d\n", chs, ld);
        return;
    }

    if ((fd = ldopen(&setup->dev, ld)) == -1) {
        warn("can't open Ldisk %d\n", ld);
        return;
    }
    ldreadck(fd, bno);                  /* perform read check */
    close(fd);
}

/*---------------------------------------------------- ldwrite() -------------
/ perform write on given 'chs (cylinder/head/sector)'
/---------------------------------------------------------------------------*/
ldwrite(setup, chs)
REG struct setup *setup;
{
    int fd, ld, bno;

    bno = chs_to_ldb(setup, chs);       /* convert to block address */
    ld = (bno >> 24) & 0x1f;            /* which logical disk */
    if (ld == LOGDR + 1)                /* reserved area? */
        ld = -1;                        /* yes */

    xprintf("ldwrite($%x, %d:$%x)\n", chs, ld, bno);

    if (bno < 0 || ldb_to_chs(setup, ld, bno &= 0xffffff) != chs) {
        warn("$%x is skipped in Ldisk %d\n", chs, ld);
        return;
    }

    if ((fd = ldopen(&setup->dev, ld)) == -1) {
        warn("can't open Ldisk %d\n", ld);
        return;
    }
    ldwriteck(fd, bno);                 /* perform write check */
    close(fd);
}

/*----------------------------- End of dsetldio.c --------------------------*/
