/*----------------------------------------------------------------------------
/ dset1.c - a part of 'dsetup'
/
/---------------------------------------------------------------------------*/

#include "disksect.h"
#include "dsetup.h"


struct _chs {                           /* 'chs' address */
    ushort cyl;                         /* cylinder */
    uchar  head;                        /* head */
    uchar  sec;                         /* sector */
};

extern int scsi_dtc;                    /* SCSI */


/*---------------------------------------------------- setdisk() -------------
/ collects all necessary disk information:  sector 0, bad block list, spare
/ block list,
/---------------------------------------------------------------------------*/
setdisk(setup, dev)
register struct setup *setup;
struct icbdev *dev;     /* device address of disk to operate on */
{
    struct sector0 *pd;
    register struct confld *ld;
    register struct logtype *logdrive;
    register struct badtrck *skipt;

    memset((char *) setup, 0, sizeof (*setup));

    setup->dev = *dev;                  /* read sector 0 */
    readphys0(setup);                   /* setup->memPhysdisk */
    pdfstat(setup->fd, &setup->pdstat);
    pd = PD0(setup);                    /* pd = setup->memPhysdisk->p */
    setup->cTotalSect = pd->cyldisk * pd->headcyl * pd->sechead;
    getbadlist(setup);                  /* setup->memBad */
    getsslist(setup);                   /* setup->memSpare, setup->memSkip */
    skipt = (struct badtrck *) setup->memSkip->p;

    if (checknewsetup(setup)) {         /* freshly formatted */
        setup->flag = SET_USED | SET_NEW;
        setup->cSpareGuess = /* this is somewhat bogus, sizes can change */
            roundtrack(setup, setup->cTotalSect / FRACTIONBAD);
        configit(setup, gConf[0].pconf);
    }
    else {                              /* setup current information */
        int i, avail = 0;
        int countskip;
        int counttrack;

        setup->flag = SET_USED | SET_OLD;
        ld = &setup->confRsv;           /* reserved area */
        ld->flag = CFLD_OLD | CFLD_PROTECT;
        ld->type = LD_USERSPARE;
        /* start is 0 */
        ld->cSect = pd->rv_size;        /* sector counts */
        ld->cBad.cSkip = pd->rv_skip;   /* skip sector counts */
        ld->cAssignSect = ld->cSect;    /* final number of sectors */
        countskip = ld->cBad.cSkip;
        counttrack = 0;
        ld = setup->confld;             /* first logical disk (0) */
        logdrive = pd->logdrive;        /* logical disk type */
        for (i = 0; i < LOGDR; i++, ld++, logdrive++) {
            ld->type = logdrive->ld_type;
            while (VALCHS(skipt++->cyl))
                ld->cBad.cTrack++;
            if (ld->type) {
                if (i >= pd->pd_ldnum)  /* too many logical disks */
                    errxit("funny format, some ld past ldnum");/* run dsck */
                ld->flag = (CFLD_OLD | CFLD_PROTECT);
                ld->logdriveindex = i;  /* remember position in old table */
                ld->start = logdrive->ld_strt;
                ld->cSect = logdrive->ld_size;
                ld->cAssignSect = ld->cSect;
                ld->cBad.cSkip = logdrive->ld_skip;
                ld->cSpareLim = calcspare(ld);
                avail += ld->cSect;     /* total sector counts */
                countskip += ld->cBad.cSkip;
                counttrack += ld->cBad.cTrack;
            }
        }
        while (VALCHS(skipt++->cyl))    /* reserved area skips */
            setup->confRsv.cBad.cTrack++;
        counttrack += setup->confRsv.cBad.cTrack;
        setup->cAvail = avail;
        setup->cSpareGuess = setup->cTotalSect - (avail + setup->confRsv.cSect);
        setup->cDiskBad.cSkip = countskip;
        setup->cDiskBad.cTrack = counttrack;
    }
}

/*---------------------------------------------------- configit() ------------
/ take 'conf' and apply it to the physical disk size.
/ watch out for the last disk.
/ if there is only one disk then its done by default.
/ this routine assumes an honest conf.
/---------------------------------------------------------------------------*/
configit(setup, conf)
register struct setup *setup;
register struct defconf *conf;
{
    register int i, c, *p;
    register struct confld *d;
    register int tsize;

    if (!setup->memPhysdisk)
        error("no physical disk selected");

    setup->defaultldtype = conf->normaltype;
    d = &setup->confRsv;
    d->flag &= ~CFLD_PROTECT;
    d->type = LD_USERSPARE; /* spare treatment is no holes */
    d->cSect = roundtrack(setup, conf->cRsv);
    d->cSpareLim = d->cSect; /* no holes */
    setup->cAvail = setup->cTotalSect - (d->cSect + setup->cSpareGuess);
    memset(setup->confld, 0, sizeof (struct confld) * LOGDR);

    /* set up a big logical disk */
    d = setup->confld;
    setupldtype(setup, d, setup->cAvail, conf->normaltype);
    if (conf->definedcount == 0)
        return(1);

    c = setup->cAvail;
    for (i = 0, p = conf->defined; i < conf->definedcount; i++, d++, p += 2) {
        setupldtype(setup, d, roundtrack(setup, p[0]), p[1]);
        c -= d->cSect;
        if (c < 0)
            return(0); /* oops, won't fit */
    }

    d--; /* point to the last logical disk */
    if (c == 0)
        goto exitok; /* it was lucky to just fit, probably a bug */

    if (c < MIN_LD_SIZE || i==LOGDR) {
        warn("last disk of selected configuration extended");
        d->cSect += c;
        goto exitok;
    }

    if (conf->normal == -1 || c < conf->normal) { /* make the remains one disk */
        if (c < conf->normal)
            warn("last disk is smaller than expected");
        i++;
        (++d)->cSect = c;
        d->type = conf->normaltype;
        goto exitok;
    }

    /* fill up the rest of the disk, the last will be the biggest */
    while (c >= (tsize = roundtrack(setup, conf->normal)) && i != LOGDR) {
        setupldtype(setup, ++d, tsize, conf->normaltype);
        c -= d->cSect;
        i++;
    }
    /* add in the remains */
    d->cSect += c;

exitok:
    setupld(setup, d, d->cSect);
    return(1);
}

checkldi0(i)
{
    if (i < 0 || i >= LOGDR)
        error("bad numeric for logical disk %d",i);
}

checkldi(i)
{
    if (i < -1 || i >= LOGDR )
        error("bad numeric for logical disk %d",i);
}

modifyldtype(setup,i,t)
struct setup *setup;
{
    register struct confld *ld;

    checkldi(i);
    ld = &setup->confld[i];
    if (!ld->type)
        error("cannot set type of empty ld");
    setupldtype(setup, ld, ld->cSect, t);
    checkusersetup(setup, warn, errorbuf);
}

setinodesize(setup, i, s)
struct setup *setup;
{
    register struct confld *ld;

    checkldi(i);
    ld = &setup->confld[i];
    if (ld->flag & CFLD_PROTECT)
        error(" cannot modify protected disk");

    switch (ld->type) {
        case LD_UNIXFS:
        case LD_USERSPLIT:
            ld->cSpareLim = s;
            break;
        default:
            error("can only modify clean (inode) size on FSYS and SPLIT types");
    }

    checkusersetup(setup, warn, errorbuf);
}

/*---------------------------------------------------- setupwritei() ---------
/ if logical disk 'i' is ok to write to then return its ld pointer
/---------------------------------------------------------------------------*/
struct confld *setupwritei(setup,i)
struct setup *setup;
register int i;
{
    register struct confld *ld;

    checkldi(i);
    ld = &setup->confld[i];
    if (ld->flag & (CFLD_PROTECT|CFLD_FREEZE))
        error("logical disk %d is marked unmodifiable", i);
    return(ld);
}

/*---------------------------------------------------- writelast() -----------
/ find the last disk of contiguous non protected disks which can be modified.
/ Go to the end of non-protected according to flag.  Then pick the last
/ according to flag.
/---------------------------------------------------------------------------*/
writelasti(setup,i,flag)
register struct setup *setup;
register int i;
{
    register struct confld *ld;
    register int istart;

    checkldi(i);
    ld = &setup->confld[i];
    istart = i;
    if (ld->flag & CFLD_PROTECT)
        error("cannot modify logical disk %d",i);
    if (!ld->type && (flag & NOPASS_EMPTY) )
        return(i);

    /* get to the end of the non-protected ld's */
    for (; i < LOGDR; i++, ld++)
        if ( ld->type && (ld->flag & CFLD_PROTECT)
        ||  !ld->type && (flag & NOPASS_EMPTY) )
            break;

    /* now backup, if we get past istart it's an error */
    for (i--, ld--; i >= istart; i--, ld--)
        if ( !ld->type && (flag & EMPTY_OK)
          ||  ld->type && (ld->flag & CFLD_FREEZE) && (flag & FREEZE_OK)
          ||  ld->type && !(ld->flag & CFLD_FREEZE))
            break;

    if (istart > i)
        error("cannot modify logical disk %d",istart);
    return(i);
}

/*---------------------------------------------------- insertld() ------------
/ create empty logical disk at 'i'
/ make sure there's room and move everything down
/---------------------------------------------------------------------------*/
insertld(setup, i)
register struct setup *setup;
{
    register struct confld *ld0;
    struct confld *srcld;

    if (i == LOGDR)
        error("cannot insert at end of logical drive table");
    if (i < 0 || i > LOGDR-1)
        error("non existent logical disk %d",i);
    ld0 = setup->confld;

    /* find the first empty after 'i' */
    for (srcld = &ld0[i]; srcld < &ld0[LOGDR] && srcld->type; srcld++)
        ;
    if (srcld == &ld0[LOGDR])
        error("no room for new entry");
    srcld--;

    for (; srcld >= &ld0[i]; srcld--)
        srcld[1] = *srcld;
    memset(&ld0[i], 0, sizeof (ld0[i]));
}

/*---------------------------------------------------- removeld() ----------*/
removeld(setup,i)
register struct setup *setup;
{
    struct confld *dstld,*nextld;
    int got_one = 0;
    unsigned char temp_flag;

    checkldi(i);

    dstld = &setup->confld[i];
    if (dstld->type)
        software_bug("removeld", "Can only remove empty drive");

    while (dstld < &setup->confld[LOGDR]) {
        if (!(nextld = (dstld+1))->type && got_one)
            break;
        if (nextld->type)
            got_one++;
        temp_flag = dstld->flag;
        *dstld = *nextld;
        dstld->flag = temp_flag;
        dstld++;
        memset(dstld, 0, sizeof (*dstld));
    }
}

setldsize(setup,inew,size,islack)
register struct setup *setup;
{
    register struct confld *ldnew, *ldslack;
    int lo,hi;
    int slackadjust;                    /* >0, add to ldslack, <0, remove from ldslack */
    int endcontig;

    /* three cases
     * - some disk in the middle
     * - the last disk (may be like previous case)
     * - the disk after the last disk
     */
    size = roundtrack(setup, size);
    if (inew != RV_LOGDR) {
        ldnew = setupwritei(setup,inew);
        if (inew == islack) {           /* special case, split an existing disk */
            insertld(setup, islack = inew + 1); /* make an empty ld after i */
        }
    }
    else {
        ldnew = &setup->confRsv;
        setup->cAvail = setup->cTotalSect - (size + setup->cSpareGuess);
    }
    ldslack = (islack != -1) ? setupwritei(setup,islack) : 0;

    if (!ldslack) {                     /* take the last disk of the sequence */
        islack = writelasti(setup,inew+1,NOPASS_EMPTY|NOFREEZE_OK);
        ldslack = &setup->confld[islack];
        if (!ldslack->type) {           /* currently not used */
            memset(ldslack, 0, sizeof (*ldslack));
        }
    }
    else {
        /* check contiguous writeable logical disks between new,slack */
        if (inew < islack) {
            lo = inew;
            hi = islack;
        }
        else {
            lo = islack;
            hi = inew;
        }

        endcontig = writelasti(setup,lo+1,PASS_EMPTY|FREEZE_OK|EMPTY_OK);
        if (endcontig < hi)
            error("non contiguous, unprotected logical drives");
    }
    if (!ldslack->type)
        ldslack->type = ldnew->type;

    if (ldnew->type) { /* adjust an existing disk */
        slackadjust = ldnew->cSect - size;
    }
    else { /* create a logical disk */
        ldnew->type = setup->defaultldtype;
        ldnew->flag = 0;
        slackadjust = -size;
    }
    setupld(setup,ldnew,size);
    /* adjust the slack disk */
    setupld(setup,ldslack,ldslack->cSect+slackadjust);
    checkusersetup(setup,warn,errorbuf);
    /*
     *  the following would be nice in some form
     */
    /*  if a logical disk is made larger and no slack disk
     *  is specified then absorb the following logical disks
     *  until no more room is left and then absorb previous logical
     *  disks.
     */
    /*
     *  when absorbing the following or previous collapse
     *  do not collapse the table.  may want to do this.
     */
}

checkusersetup(setup, reportproc, escapebuf)
register struct setup *setup;
jmp_buf escapebuf;
int (*reportproc)();
{
    int i;
    register struct confld *ld;
    register int check, allok;
    int tsum;
    register struct sector0 *pd;
    int countSWAP = 0;
    int countPWF = 0;

    pd = PD0(setup);
    check = 0;
    allok = 1;
    /* check out reserved first */
    ld = &setup->confRsv;
    tsum = pd->sechead * 3;
    if (ld->cSect < tsum) {
        reperr(reportproc,"reserved are size too small, at least %d", tsum);
        allok = 0;
    }
    if (ld->cSect > MAX_SIZE_RV) {
        reperr(reportproc,"reserved size too big, %d max", MAX_SIZE_RV);
        allok = 0;
    }
    for (i = 0, ld = setup->confld; i < LOGDR; i++, ld++)
        if (ld->type) {
            int tlim;

            if (ld->type == LD_SWAP)
                countSWAP++;
            else if (ld->type == LD_PWF)
                countPWF++;
            if (ld->cSect < MIN_LD_SIZE) {
                reperr(reportproc, "check: ld %d, bad size %d",i,ld->cSect);
                allok = 0;
            }
            if (ld->cSect % pd->sechead) {
                reperr(reportproc, "check: ld %d, bad size modulo %d",
                        i, ld->cSect);
                allok = 0;
            }
            tlim = calcspare(ld, setup);
            if (ld->cSpareLim > ld->cSect) {
                warn("check: ld %d, impossible clean (inode) size %d", i, tlim);
                ld->cSpareLim = ld->cSect;
            }
            if (!(ld->flag & CFLD_PROTECT))
                switch(ld->type) {
        	    case LD_USERSPLIT:
                    case LD_UNIXFS:
                        if (ld->cSpareLim != tlim)
                            warn("non standard clean/inode bounds on logical disk %d",
                                    i);
                        break;
                    default:
                        if (tlim != ld->cSpareLim) {
                            allok = 0;
                            reperr(reportproc,
                                    "ld type %s cannot have modified clean/inode bounds",
                                    ascldtype[ld->type]);
                        }
                }
            check += ld->cSect;
        }
    if (countSWAP > 1)
        warn("at most one SWAP area per disk will be allowed");
    if (countPWF > 1)
        warn("at most one PWRF area per disk will be allowed");
    if (setup->cAvail != check) {
        allok=0;
        reperr(reportproc,"available sectors (%d) != ld sizes (%d)",
                setup->cAvail,check);
    }
    if (setup->cTotalSect !=
            (tsum = pd->cyldisk
                  * pd->headcyl
                  * pd->sechead)) {
        allok = 0;
        reperr(reportproc,"total sect (%d) != physdisk sectors (%d)",
                setup->cTotalSect, tsum);
    }
    if (setup->cTotalSect !=
            (tsum = check
                  + setup->confRsv.cSect
                  + setup->cSpareGuess)) {
        allok = 0;
        reperr(reportproc,"total sect (%d) != ld+rsv+spare (%d)",
                setup->cTotalSect, tsum);
    }
    if ((!allok) && escapebuf)
        longjmp(escapebuf,1);
    return(allok);
}

printphysdisk(setup)
struct setup *setup;
{
    register struct sector0 *pd = PD0(setup);
    register int temp;

    temp = setup->dev.dev_num;
    printf("Controller %d, Drive %d  =  c%dd%d\n",
            (temp & ~3) >> 4, temp & 3, (temp & ~3) >> 4, temp & 3);
    printf("Flag = $%x", setup->flag);
    xprintf(" ==> ");
    if (setup->flag & SET_USED)  xprintf("Used ");
    if (setup->flag & SET_WRIT)  xprintf("Written ");
    if (setup->flag & SET_FIG)   xprintf("Fig ");
    if (setup->flag & SET_NEW)   xprintf("New ");
    if (setup->flag & SET_OLD)   xprintf("Old ");
    if (setup->flag & SET_MOD)   xprintf("Mod ");
    if (setup->flag & SET_WBADS) xprintf("WBadS");

    printf("\n%d cylinders, %d heads, %d sectors/head\n",
            pd->cyldisk, pd->headcyl, pd->sechead);
    printf("%d sectors in the reserved area, %d sectors for sparing\n",
            setup->confRsv.cSect, setup->cSpareGuess);
    printf("%d total sectors with %d sectors available\n",
            setup->cTotalSect, setup->cAvail);
    if (setup->flag & SET_FIG) {
        register struct cbad *cb = &setup->cDiskBad;

        printf("%d tracks w/bad, %d holes, %d spares, %d skips\n",
                cb->cTrack, cb->cHole, cb->cSpare, cb->cSkip);
    }
}

printsetup(setup)
register struct setup *setup;
{
    register int i;
    register struct confld *ld;

    printf("\n  %d sectors available\n", setup->cAvail);
    printf(" ld type st    size   clean\n");
    for (i = 0, ld = setup->confld; i < LOGDR; i++, ld++)
        if (ld->type) {
            printld(i, ld);
        }
    checkusersetup(setup,warn,0);
}

printld(i, ld) /* ADD VERBOSE FLAG */
struct confld *ld;
{
    printf(" %2d %4s ", i, ascldtype[ld->type]);
    if (ld->flag & CFLD_PROTECT)
        printf(" P");
    else if (ld->flag & CFLD_FREEZE)
        printf(" F");
    else
        printf("  ");
    printf(" %7d  %6d\n", ld->cSect, ld->cSpareLim);
}

setupldtype(setup, ld, size, type)
register struct confld *ld;
{
    ld->type = type;
    setupld(setup, ld, size);
}

setupld(setup, ld, size)
struct setup *setup;
register struct confld *ld;
{
    if (!size) {
        memset(ld, 0, sizeof (*ld));
        return;
    }
    if (&setup->confRsv != ld && size < MIN_LD_SIZE) {
        error("attempt to set ld %d to %d sectors (%d is smallest)",
                ld - setup->confld, size, MIN_LD_SIZE);
    }
    ld->cSect = ld->cAssignSect = size;
    ld->cSpareLim = calcspare(ld, setup);
    if (!ld->cSect) {
        ld->type = 0;
        ld->cSpareLim = 0;
    }
}

calcspare(ld, setup)
register struct setup *setup;
register struct confld *ld;
{
    register int n;
    register int size;

    if (setup->flag & SET_FIG)
        size = ld->cAssignSect;
    else
        size = ld->cSect;

    switch (ld->type) {
        case LD_SWAP:
        case LD_USERSKIP:
        case LD_USERHOLE:
            return(0);
        case LD_UNIXFS:
        case LD_USERSPLIT:
            n = size / INODEFACT;
            if (n <= 0)
                n = 1;
            if (n > MKFS_MAGIC / NBINODE)
                n = MKFS_MAGIC / NBINODE;
            return(n + 2);
        case LD_USERSPARE:
        case LD_PWF:
            return(size);
    }
    software_bug("calcspare", "Unknown disk type - %d", ld->type);
}

/*---------------------------------------------------- block_to_ch() ---------
/ converts block address to ch (cylinder, head) address
/---------------------------------------------------------------------------*/
block_to_ch(setup,bno)
register int bno;
register struct setup *setup;
{
    register struct _chs *nchs;
    register struct sector0 *pd;
    uint ret;

    nchs = (struct _chs *) &ret;
    pd = PD0(setup);
    nchs->cyl = bno / pd->seccyl;
    nchs->head = (bno % pd->seccyl) / pd->sechead;
    nchs->sec = 0;
    return(ret);
}

/*---------------------------------------------------- block_to_chs() --------
/ converts block address to chs (cylinder, head, sector) address
/
/ a 'chs' is an 32-bit unsigned integer in the following format:
/
/   msb                          lsb
/    3         2         1
/   10987654321098765432109876543210
/   <-------c------><---h--><---s-->
/
/   struct _chs {
/       ushort cyl;         which cylinder
/       uchar head;         which head
/       uchar sector;       which sector
/   };
/---------------------------------------------------------------------------*/
block_to_chs(setup, bno)
register struct setup *setup;
register int bno;
{
    register struct _chs *n;
    register uint hs;
    register struct sector0 *pd;
    uint ret;

    pd = PD0(setup);
    n = (struct _chs *) &ret;
    n->cyl = bno / pd->seccyl;
    hs = bno % pd->seccyl;
    n->head = hs / pd->sechead;
    n->sec = hs % pd->sechead;
    return(ret);
}

/*---------------------------------------------------- chs_to_block() --------
/ converts 'chs' to block address where a block is defined by BLKSIZE
/---------------------------------------------------------------------------*/
chs_to_block(setup,chs)
register struct setup *setup;
{
    register struct _chs *p;
    register uint ret;
    register struct sector0 *pd;

    pd = PD0(setup);
    p = (struct _chs *) &chs;
    ret = (p->cyl * pd->seccyl) + (p->head * pd->sechead) + p->sec;
    return(ret);
}

/*---------------------------------------------------- roundtrack() ----------
/ rounds up track number to even sechead counts
/---------------------------------------------------------------------------*/
roundtrack(setup, sz)
register struct setup *setup;
{
    register int rem, tsechead;

    if (sz < 0)
        error("Unusable track size %d", sz);
    tsechead = PD0(setup)->sechead;
    rem = sz % tsechead;
    if (rem)
        sz += (tsechead - rem);
    return(sz);
}

/*---------------------------------------------------- chs_to_ldb() ----------
/ converts 'chs' to logical disk block address
/---------------------------------------------------------------------------*/
chs_to_ldb(setup, chs)
register struct setup *setup;
{
    register int bno;
    int i, ld, sh;
    struct confld *lp, *le;
    struct badtrck *bp, *be;

    bno = chs_to_block(setup, chs);
    le = &setup->confld[LOGDR];
    sh = PD0(setup)->sechead;
    for (lp = &setup->confRsv; lp < le; lp++) {
        if (lp->type && bno < lp->start + lp->cAssignSect + lp->cBad.cSkip*sh)
            break;
    }
    bno -= lp->start;
    if ((ld = lp - setup->confld) == -1)
        ld = LOGDR + 1;
    bp = (struct badtrck *)setup->memSkip->p;
    be = &bp[setup->cMemSkip];
    for (i = 0; i < ld && bp < be; bp++) {
        if (VALCHS(*bp) == 0)
            i++;
    }
    chs = (chs & 0xffff0000) + ((chs & 0xff00) >> 8);
    for ( ;bp < be && VALCHS(*bp) && VALCHS(*bp) < chs; bp++)
        bno -= sh;
    if (bno < 0)
        bno = 0x8000000 - bno;
    return( (ld << 24) + bno );
}

/*---------------------------------------------------- ldb_to_chs() ----------
/ converts logical disk block address (ldb) to a 'chs'
/---------------------------------------------------------------------------*/
ldb_to_chs(setup, ld, bno)
struct setup *setup;
{
    int ch, i;
    struct badtrck *bp, *be;

    if (ld < -1 || ld >= (int) PD0(setup)->pd_ldnum)
        return(-1);
    bno += setup->confld[ld].start;
    bp = (struct badtrck *) setup->memSkip->p;
    be = &bp[setup->cMemSkip];
    for (i = 0; i < ld && bp < be; bp++) {
        if (VALCHS(*bp) == 0)
            i++;
    }
    ch = block_to_chs(setup, bno);
    ch = (ch & 0xfff0000) + ((ch >> 8) & 0xff);
    for ( ; bp < be && VALCHS(*bp) && VALCHS(*bp) <= ch; bp++) {
        bno += PD0(setup)->sechead;
        if ((++ch & 0xff) == PD0(setup)->headcyl)
            ch = (ch & 0xfff0000) + 0x10000;
    }
    return(block_to_chs(setup, bno));
}

/*------------------------------- End of dset1.c ---------------------------*/
