/*----------------------------------------------------------------------------
/ scsimain.c - SCSI portion of disktest - main module
/
/ Craig J. Kim      June 1988
/ (c) Copyright 1988 ARIX Corp.
/---------------------------------------------------------------------------*/

#include "scsidt.h"

/*---------------------------------------------------- scsitest() ------------
/
/---------------------------------------------------------------------------*/
scsitest()
{
    register int c, i;

initdrv:
    while (!initscsi())                 /* is disk initialized properly? */
        if (getyn("Would you like to format the drive"))
            scsi_format();
        else {
            if (debugflag)
                if (getyn("Would like to initialize sector 0 and 1")) {
                    if (!rewrite_sec01())
                        printf("  Initializing sector 0 aborted!\n");
                    continue;
                }
            printf("Test aborted\n\n");
            return;
        }

    printscsitype(&MthrBlk);            /* print disk physical properties */
    AltPat = FALSE;
    DBPat = FALSE;
    printf("\n");
    while (1) {
        setjmp(MenuBuffer);             /* setup error trap */
        SoftDiskErr = 0;                /* init for new test */
        HardDiskErr = 0;                /* hard error counts */
        CompErr = 0;                    /* compare error count */
        Pass = 1;                       /* how many tests done thus far */
        TotEntries = 0;                 /* bad sector entry count */
        if (DBPat)
            Pattern = 0;                /* initialize the read/write pattern */
        else if (!AltPat)
            Pattern = 0xdbdbdbdb;       /* default pattern */
        ErrRootPtr = ErrListPtr = ErrData;
        Compare = FALSE;                /* init */

        printf(Prompt);
        if ((c = getreply()) < 0)
            return;

        if (isupper(c))
            c = _tolower(c);
        switch (c) {
            case 'a':                   /* change parameters */
                scsi_params();
                break;
            case 'b':                   /* read test */
                q_comp_dump();
                p_instr();
                while (1) {
                    if (!scsi_test_drive(BeginSector, NumOfSectors, PREADPS))
                        break;
                    ++Pass;
                }
                break;
            case 'c':                   /* write test */
                p_instr();
                while (1) {
                    if (!scsi_test_drive(BeginSector, NumOfSectors, PWRITPS))
                        break;
                    ++Pass;
                }
                break;
            case 'd':                   /* read/write test */
                q_comp_dump();
                p_instr();
                while (1) {
                    if (!scsi_test_drive(BeginSector, NumOfSectors, PWRITPS))
                        break;
                    if (!scsi_test_drive(BeginSector, NumOfSectors, PREADPS))
                        break;
                    ++Pass;
                    if (!AltPat) {      /* do not shift an alternate pattern */
                        i = Pattern;
                        Pattern <<= 1;
                        if (i & 0x80000000)
                            Pattern++;
                    }
                }
                break;
            case 'e':                   /* change pattern */
                DBPat = FALSE;
                AltPat = FALSE;
                if (getyn("Use the debug patterns")) {
                    DBPat = TRUE;
                    break;
                }
                if (getyn("Use the default pattern"))
                    break;
                Pattern = get_num("Enter test pattern");
                AltPat = TRUE;
                break;
            case 'f':                   /* format drive */
                scsi_format();
                goto initdrv;
#ifdef FUTURE
            case 'g':                   /* display bad sectors */
                disbad();
                break;
#endif
            case 'l':                   /* display physical properties */
                printscsitype(&MthrBlk);
                break;
            case 'z':                   /* toggle debug flag */
                flipdebug();
                break;
            case 13:                    /* forgive him/her */
            case '?':
                printf("\nSCSI Disk Diagnostic Menu  ver %d.%d\n", version, release);
                printf("\n");
                printf("  a   Change Parameters -- Current drive c%dd%d start 0x%x length 0x%x\n",
                    UnitNum >> 4, DriveNum, BeginSector, NumOfSectors);
                printf("  b   Continuous Read\n");
                printf("  c   Continuous Write\n");
                printf("  d   Read/Write Test\n");
                printf("  e   Alter Pattern -- Current = ");
                if (DBPat)
                    printf("DEBUG\n");
                else
                    printf("$%8x\n", Pattern);
                printf("  f   Format Drive\n");
#ifdef FUTURE
                printf("  g   Display Bad Sector List\n");
#endif
                printf("  l   List Physical Disk Information\n");
                printf("[Esc] Quit\n\n");
                break;
            case '$':
                if (debugflag) {
                    hidden_commands();
                    break;
                }
            default:
                printf(BadCmd);
                break;
        }
    }
}

hidden_commands()
{
        register int c;

        printf("\n\nWARNING: Little or NO checking is done for the following commands.\n");
        printf("Please exercise extreme caution!!  Remember, you've been warned!\n");
        while (1) {
            printf("<$>%s", Prompt);
            if ((c = getreply()) < 0)
                return;

            if (isupper(c))
                c = _tolower(c);
            switch (c) {
                case 'i':
                    if (getyn("Initialize sectors 0, 1, and 8 to default"))
                        if (!rewrite_sec01())
                            printf("Writing sector 0 aborted.\n");
                    break;
                case 'l':
                    p_logical_disks();
                    break;
                case 'r':
                    read_blocks();
                    break;
                case 's':
                    p_modesense();
                    break;
                case 'w':
                    write_blocks();
                    break;
                case 'z':                   /* toggle debug flag */
                    flipdebug();
                    break;
                case 13:                    /* forgive him/her */
                case '?':
                    printf("\n--- SCSI Disk Diagnostic Special Menu ---\n");
                    printf("\n");
                    printf("  i   Reinitialize sectors 0, 1, and 8\n");
                    printf("  l   Display logical disks\n");
                    printf("  r   Read blocks\n");
                    printf("  s   Read mode sense pages\n");
                    printf("  w   Write blocks\n");
                    printf("  z   Toggle debug flag\n");
                    printf("[Esc] Quit\n");
                    break;
                default:
                    printf(BadCmd);
                    break;
            }
        }
}

/*---------------------------------------------------- initscsi() ------------
/ determines whether or not drive has been initialized properly
/----------------------------------------------------------------------------*/
static initscsi()
{
    register int i;
    int retcode;

    printf("\nWait.  Checking the disk...");
    autosize = 0;

    retcode = scsi_init();              /* see if disk is ready and available */

    scsi_drvtype = 0;                   /* unknown */
    for (i = 0; i < SCSI_NUMDRVS; i++)
        if (!strncmp(scsi_drv[i].diskname, inquiry.idvendor, scsi_drv[i].namelen)) {
            scsi_drvtype = i;
            break;
        }

    if (retcode == FALSE)
        goto abortchecking;

    if (scsi_drv[scsi_drvtype].initfunc) {
        xprintf("Executing drive specific initialization function.\n");
        if (!(*scsi_drv[scsi_drvtype].initfunc)())
            goto abortchecking;
    }

    xprintf("Reading sector 0\n");
                                        /* read sector 0 */
    if (retcode = sc_read(&MthrBlk, 0, 1)) {
        printf("\nCannot read sector 0.\n");
        DBG(printf(">retcode = $%04x - %s\n", retcode, sc_derrmsg(retcode));)
        goto abortchecking;
    }

    /*-----------------------------------------------------------------
    / check sector 0
    /
    / if ( the mother block id is not "INIT" ) or
    /    ( the version (low byte of data member) is 0xFF ) or
    /    ( the number of logical drives ( slices ) > LOGDR (16) ) or
    /    ( the number of cyls/disk > 7FF (2047) ) {
    /	NOTE: Tue Nov  8 10:56:16 PST 1994 (hanna)
    /		according to REE, this last test is invalid for very large
    /		drives that have greater than 2047 cyls/disk. 
    /   sector 0 does not contain ARIX formatting information
    /   return(FALSE)
    / }
    /
    /----------------------------------------------------------------*/

    if (wrdcmp("INIT", MthrBlk.id) != 0 || MthrBlk.date[0] == 0xff
            || MthrBlk.pd_ldmaxnum > LOGDR /* || MthrBlk.cyldisk > 0x7ff */) {
        if (MthrBlk.date[0] != 0xff)
            MthrBlk.id[0] = 0;
        printf("\nSector 0 NOT initialized!\n");
        goto abortchecking;
    }

                                        /* read sector 1 */
    xprintf("Reading sector 1\n");
    if (retcode = sc_read(TrackBuffer, 1, 1)) {
        printf("\nCannot read sector 1.\n");
        DBG(printf(">retcode = $%04x - %s\n", retcode, sc_derrmsg(retcode));)
        MthrBlk.id[0] = 0;
        goto abortchecking;
    }

    /*----------------------------------------------------------------
    / Compare sector 0 to sector 1 - sector 1 is mirror of sector 0
    /---------------------------------------------------------------*/

    if ((i = comp_bufs(&MthrBlk, TrackBuffer, sectsiz)) < sectsiz) {
        MthrBlk.id[0] = 0;
        printf("\nSectors 0 and 1 are NOT equal at offset $%x.\n", i);
        goto abortchecking;
    }

    i = MthrBlk.sechead * MthrBlk.bytesec;
    if (i > TBsize) {
        printf("Disk too large to test.  Out of memory.\n");
        goto abortchecking;
    }

    /*------------------------------------------------
    / All is well.  Inform HUMAN of DEVICE TYPE.
    /-----------------------------------------------*/

    BeginSector = 3 * MthrBlk.sechead;  /* default-all but first 3 tracks */
    NumOfSectors = capacity.lbaddr - BeginSector;
    WrdsPerSect = MthrBlk.bytesec >> 1; /* divide by 2 */

    DBG(printf(">BeginSector = %d\n", BeginSector);  )
    DBG(printf(">NumOfSectors = %d\n", NumOfSectors);)

    return(TRUE);                       /* a success! */

abortchecking:
    printf("\n");
    return(FALSE);
}

/*---------------------------------------------------- scsi_params() ---------
/ change current testing parameters including drive unit
/---------------------------------------------------------------------------*/
scsi_params()
{
    register unsigned temp;
    register char c;

    temp = UnitNum;                     /* save */
    while (1) {
        if (!getdrive())                /* get new drive spec */
            return(1);
        if (UnitNum == temp)
            break;
        if (initscsi())
            return;
	else{
            if (getyn("Would you like to format the drive")){
            	scsi_format();
		initscsi();
	        return;
	    }
	}
    }

    if (getyn("Test all but cylinder 0")) {
        BeginSector = MthrBlk.seccyl;   /* protect cylinder 0 */
        NumOfSectors = capacity.lbaddr - BeginSector;
        /* NumOfSectors = (MthrBlk.cyldisk - 1) * MthrBlk.seccyl; */
    }
    else if (MthrBlk.pd_ldnum && getyn("Test a logical disk")) {
        while (1) {
            temp = get_num("Enter logical disk number");
            if (temp >= MthrBlk.pd_ldnum) {
                printf("Logical disk %d does not exist.  Try again.\n", temp);
                continue;
            }
            BeginSector = MthrBlk.logdrive[temp].ld_strt;
            NumOfSectors = MthrBlk.logdrive[temp].ld_size;
            return;
        }
    }
    else {
        while(1) {
            temp = get_num("Enter starting block");
            if (temp > MthrBlk.cyldisk) {
                pr_no_good();
                continue;
            }
            BeginSector = temp;
            break;
        }

        if (getyn("Test to end of drive")) {
            NumOfSectors = MthrBlk.cyldisk * MthrBlk.seccyl - BeginSector;
            return;
        }

        while (1) {
            temp = get_num("Enter number of blocks to be tested");
            if ((BeginSector + temp) > (MthrBlk.cyldisk * MthrBlk.seccyl))
                printf("Exceeds size of drive.  Try again.\n");
            else {
                NumOfSectors = temp;
                break;
            }
        }
    }
}

/*---------------------------------------------------- printscsitype() -------
/ printscsitype()
/
/ print the disk manufacturer name, and layout
/----------------------------------------------------------------------------*/
static printscsitype(sp)
register struct sector0 *sp;
{
    register int secdisk;

    printf("\n");
    printf("        Drive Name: %s%s\n", inquiry.idvendor, scsi_drvtype ? "" : " (Uncertain)");
    printf("    Unit Specifier: c%dd%d\n", UnitNum >> 4, DriveNum);
    printf("    # of Cylinders: %d ($%x)\n", sp->cyldisk, sp->cyldisk);
    printf("        # of Heads: %d ($%x)\n", sp->headcyl, sp->headcyl);
    printf("     Sectors/Track: %d ($%x)\n", sp->sechead, sp->sechead);
    secdisk = sp->sechead * sp->headcyl * sp->cyldisk;
    printf("Total Sectors/Disk: %d ($%x)\n", secdisk, secdisk);
    printf("Avail Sectors/Disk: %d ($%x)\n", sp->secdisk, sp->secdisk);
    printf("      Bytes/Sector: %d ($%x)\n", sp->bytesec, sp->bytesec);
}

static p_logical_disks()
{
    static char *ldtype[] = { "NULL", "FSYS", "SWAP", "PWRF", "HOLE", "FSPR", "FSKP", "SPLT" };
    register int i;

    printf("\n");
    printf("Reserved area starts at sector %d and is %d sectors wide.\n\n",
            MthrBlk.rv_strt, MthrBlk.rv_size);
    if (!MthrBlk.pd_ldnum) {
        printf("No logical disks defined.\n");
        return;
    }
    printf("ld  type  start   size\n");
    printf("--  ----  ------  ------\n");
    for (i = 0; i < MthrBlk.pd_ldnum; i++)
        if (MthrBlk.logdrive[i].ld_type < 0
                || MthrBlk.logdrive[i].ld_type > sizeof (ldtype))
            printf("%2d  UNKNOWN DISK TYPE\n", i);
        else
            printf("%2d  %s  %6d  %6d\n",
                    i, ldtype[MthrBlk.logdrive[i].ld_type],
                    MthrBlk.logdrive[i].ld_strt, MthrBlk.logdrive[i].ld_size);
}

static p_instr()
{
    printf("\nDuring the test, press 's' to view current status or [Esc] to quit.\n");
}

static p_modesense()
{
    struct _scsi_mode_cache     cache;
    struct _scsi_mode_error     error;  /* MODE SELECT/SENSE */
    struct _scsi_mode_conn      conn;   /* MODE SELECT/SENSE */
    struct _scsi_mode_format    format; /* MODE SELECT/SENSE */
    struct _scsi_mode_geo       geo;    /* MODE SELECT/SENSE */
    register int c, retcode;

    while (1) {
        printf("\nMode Sense Page: C)ache,D)isconnect,E)rror,F)ormat,R)igid,Q)uit--> ");
        if ((c = getreply()) < 0)
            return;
        putchar('\n');
        if (isupper(c))
            c = _tolower(c);
        switch (c) {
            case 'c':
                if (!scsi_drv[scsi_drvtype].s_cache) {
                    printf("This drive doesn't support cache control mode sense page.\n");
                    break;
                }
                printf("Reading cache control parameters page\n");
                retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_CACHE, &cache);
                if (retcode) {
                    DBG(printf(">After MOD SEN:  $%x", retcode);            )
                    printf("\n%s\n", sc_derrmsg(errno));
                    printf("Unable to read cache control page.\n");
                }
                else
                    p_ms_cache(&cache);
                break;
            case 'd':
                if (!scsi_drv[scsi_drvtype].s_conn) {
                    printf("This drive doesn't support disconnect/reconnect mode sense page.\n");
                    break;
                }
                printf("Reading disconnect/reconnect page\n");
                retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_CONN, &conn);
                if (retcode) {
                    DBG(printf(">After MOD SEN:  $%x", retcode);            )
                    printf("\n%s\n", sc_derrmsg(errno));
                    printf("Unable to read disconnect/reconnnect page.\n");
                }
                else
                    p_ms_connect(&conn);
                break;
            case 'e':
                printf("Reading error recovery page\n");
                retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_ERROR, &error);
                if (retcode) {
                    DBG(printf(">After MOD SEN:  $%x", retcode);            )
                    printf("\n%s\n", sc_derrmsg(errno));
                    printf("Unable to read error recovery page.\n");
                }
                else
                    p_ms_error(&error);
                break;
            case 'f':
                printf("Reading format parameters page\n");
                retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_FORMAT, &format);
                if (retcode) {
                    DBG(printf(">After MOD SEN:  $%x", retcode);            )
                    printf("\n%s\n", sc_derrmsg(errno));
                    printf("Unable to read format parameters page.\n");
                }
                else
                    p_ms_format(&format);
                break;
            case 'r':
                printf("Reading rigid disk parameters page\n");
                retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_GEO, &geo);
                if (retcode) {
                    DBG(printf(">After MOD SEN:  $%x", retcode);            )
                    printf("\n%s\n", sc_derrmsg(errno));
                    printf("Unable to read rigid disk parameters page.\n");
                }
                else
                    p_ms_rigid(&geo);
                break;
            case 'q':
                return;
            default:
                printf("Invalid choice.  Please type one of \"cdefrq\" or [Esc] to quit\n");
                break;
        }
    }
}

p_ms_cache(ca)
struct _scsi_mode_cache *ca;
{
        p_ms_sense(ca);
        printf("        Cache control: $%x", ca->pcache);
        if (ca->pcache) {
            printf(" - ");
            if (ca->pcache & SCSI_M_CC_WIE)
                printf("Write-index-enabled ");
            if (ca->pcache & SCSI_M_CC_CE)
                printf("Cache-enabled ");
            if (ca->pcache & SCSI_M_CC_CTS)
                printf("Cache-table-size=%d", ca->pcache & SCSI_M_CC_CTS);
            printf("\n");
        }
        printf("Prefetch thresh hold: $%x\n", ca->thresh);
        printf("    Maximum prefetch: %d\n", ca->maxfetch);
        printf("Max prefetch multipr: %d\n", ca->maxpfm);
        printf("    Minimum prefetch: %d\n", ca->minfetch);
        printf("Min prefetch multipr: %d\n", ca->minpfm);
}

p_ms_connect(con)
struct _scsi_mode_conn *con;
{
        p_ms_sense(con);
        printf("   Buffer full ratio: %d ($%x)\n", con->bufull, con->bufull);
        printf("  Buffer empty ratio: %d ($%x)\n", con->bufmt, con->bufmt);
        printf("Bus inactivity limit: %d ($%x)\n", con->bilim, con->bilim);
        printf("Disconnect tim limit: %d ($%x)\n", con->dtlim, con->dtlim);
        printf("  Connect time limit: %d ($%x)\n", con->ctlim, con->ctlim);
}

p_ms_error(err)
struct _scsi_mode_error *err;
{
        p_ms_sense(err);
        printf("    Error correction: $%x", err->errec);
        if (err->errec) {
            printf(" - ");
            printf("%s", err->errec & SCSI_M_ER_AWRE ? "AWRE " : "");
            printf("%s", err->errec & SCSI_M_ER_ARRE ? "ARRE " : "");
            printf("%s", err->errec & SCSI_M_ER_TB   ? "TB "   : "");
            printf("%s", err->errec & SCSI_M_ER_RC   ? "RC "   : "");
            printf("%s", err->errec & SCSI_M_ER_EEC  ? "EEC "  : "");
            printf("%s", err->errec & SCSI_M_ER_PER  ? "PER "  : "");
            printf("%s", err->errec & SCSI_M_ER_DTE  ? "DTE "  : "");
            printf("%s", err->errec & SCSI_M_ER_DCR  ? "DCR"   : "");
        }
        printf("\n         Retry count: %d\n", err->retry);
        printf("     Correction span: $%x\n", err->corrspan);
        printf("   Head offset count: $%x\n", err->corrspan);
        printf("  Data strobe offset: $%x\n", err->strobe);
        printf("              tlimit: $%x\n", err->tlimit);
}

p_ms_format(fmt)
struct _scsi_mode_format *fmt;
{
        p_ms_sense(fmt);
        printf("         tracks/zone: %d ($%x)\n", fmt->tpz, fmt->tpz);
        printf("alternate sects/zone: %d ($%x)\n", fmt->aspz, fmt->aspz);
        printf("alternate traks/zone: %d ($%x)\n", fmt->atpz, fmt->atpz);
        printf("alternate traks/volm: %d ($%x)\n", fmt->atpv, fmt->atpv);
        printf("       sectors/track: %d ($%x)\n", fmt->spt, fmt->spt);
        printf("         sector size: %d ($%x)\n", fmt->secsize, fmt->secsize);
        printf("          interleave: %d\n", fmt->inter);
        printf("          track skew: %d ($%x)\n", fmt->tskew, fmt->tskew);
        printf("       cylinder skew: %d ($%x)\n", fmt->cskew, fmt->cskew);
        printf("          drive type: %d ($%x)\n", fmt->drvtype, fmt->drvtype);
}

p_ms_rigid(geo)
struct _scsi_mode_geo *geo;
{
        register int num;

        p_ms_sense(geo);
        printf("  Max # of cylinders: %d\n", a3toi(geo->cyl));
        printf("      Max # of heads: %d\n", geo->heads);
        num = a3toi(geo->scylwp);
        printf("   Write precomp cyl: %d ($%x)\n", num, num);
        num = a3toi(geo->scylrw);
        printf("   Reduced write cyl: %d ($%x)\n", num, num);
        printf("           step rate: %d\n", mgeo.dstep);
        num = a3toi(geo->land);
        printf("         landing cyl: %d ($%x)\n", num, num);
}

p_ms_sense(sen)
struct _scsi_mode_sense *sen;
{
        register int num;

        printf("         Medium type: $%x\n", sen->medtype);
        printf("       Write protect: %s\n", (sen->wp & SCSI_M_WP_WP) ? "On" : "Off");
        printf("             Density: %d ($%x)\n", sen->density, sen->density);
        num = a3toi(sen->numlb);
        printf(" # of logical blocks: %d ($%x)\n", num, num);
        num = a3toi(sen->lblen);
        printf("Logical block length: %d ($%x)\n", num, num);
}

/*--------------------------- End of scsimain.c ----------------------------*/
