/*----------------------------------------------------------------------------
/ scsiinit.c - check a scsi device
/
/ Craig J. Kim      June 1988
/ (c) Copyright 1988 ARIX Corp.
/---------------------------------------------------------------------------*/

#include "scsidt.h"
#include "scsi_error.h"

int scsi_init();                        /* check if drive is ok */
int scsi_format();                      /* format a disk */
static int initsec0();                  /* initialize sector 0 info */


/*---------------------------------------------------- scsi_init() ---------*/
int scsi_init()
{
        register int tries;
        register int retcode;

        for (tries = 0; tries < 3; tries++) {
            xprintf("\nrun #%d", tries);
            if (retcode = sc_inquiry(&inquiry)) {
                DBG(printf(">After INQUIRY:  $%x", retcode);         )
                DBG(printf(" - %s\n", sc_derrmsg(errno));    )
                goto DrvNotRdy;
            }
            /* analyse inquiry data */
            if (inquiry.devtype == 0x7f) {
                goto DrvNotRdy;
            }
            xprintf("\n%s\n", inquiry.idvendor);

            if (retcode = sc_testunit()) {
                DBG(printf(">After TESTUNT:  $%x", retcode);        )
                DBG(printf(" - %s\n", sc_derrmsg(errno));    )
                printf("\n%s\n", sc_errmsg(rerror.key & SCSI_M_SK_SKEY));
                if (retcode = sc_testunit()) {
                    DBG(printf(">After TESTUNT:  $%x", retcode);    )
                    DBG(printf(" - %s\n", sc_derrmsg(errno));)
                    if (errno != 6) {
                        if (retcode = sc_testunit()) {
                            DBG(printf(">After TESTUNT:  $%x", retcode);)
                            DBG(printf(" - %s\n", sc_derrmsg(errno));)
                            if (errno != 6) {
                                printf("\n%s\n", sc_derrmsg(errno));
                                continue;
                            }
                        }
                    }
                    else {
                        if (retcode = sc_start()) {
                            DBG(printf(">After START:    $%x", retcode);)
                            DBG(printf(" - %s\n", sc_derrmsg(errno));)
                            if (errno != 2) {
                                printf("\n%s\n", sc_derrmsg(errno));
                                continue;
                            }
                        }
                        if (retcode = sc_testunit()) {
                            DBG(printf(">After TESTUNT:  $%x", retcode);)
                            DBG(printf(" - %s\n", sc_derrmsg(errno));)
                            if (errno != 6) {
                                printf("\n%s\n", sc_derrmsg(errno));
                                continue;
                            }
                            printf("\nTimed out\n");
                            continue;
                        }
                    }
                }
            }

            if (!(retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_ERROR, &merror))) {
                if (!(retcode = sc_readcap(&capacity))) {
                    DBG(printf(">Logical block address: %d\n", capacity.lbaddr);)
                    DBG(printf(">         Block length: %d\n", capacity.blklen);)
                    if (capacity.blklen != BLKSIZE)
                        break;
                    return(TRUE);
                }
                DBG(printf(">After READ CP:  $%x", retcode);    )
                DBG(printf(" - %s\n", sc_derrmsg(errno));)
            }
            DBG(else {                                              )
            DBG(printf(">After MOD SEN:  $%x", retcode);    )
            DBG(printf(" - %s\n", sc_derrmsg(errno));)
            DBG(}                                                   )
        }
        printf("\nDisk is NOT formatted.\n");
        return(FALSE);

DrvNotRdy:
        printf("\n\nDrive Not Ready\n");
        longjmp(MenuBuffer, 1);
}

/*---------------------------------------------------- scsi_format() -------*/
int scsi_format()
{
        register int retcode;

        xprintf("Reading saved error recovery page\n");
        retcode = sc_modesense(SCSI_V_PC_SAV, SCSI_V_MS_ERROR, &merror);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read error recovery page.");
            goto format_aborted;
        }
        DBG(p_ms_error(&merror);)

        xprintf("Reading saved format parameters page\n");
        retcode = sc_modesense(SCSI_V_PC_SAV, SCSI_V_MS_FORMAT, &mformat);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read format parameters page.");
            goto format_aborted;
        }
        DBG(p_ms_format(&mformat);)

        xprintf("Reading saved rigid disk parameters page\n");
        retcode = sc_modesense(SCSI_V_PC_SAV, SCSI_V_MS_GEO, &mgeo);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read rigid disk parameters page.");
            goto format_aborted;
        }
        DBG(p_ms_rigid(&mgeo);)

        printf("Warning:  Formatting the disk erases ALL data on the entire disk!\n");
        if (!getyn("Do you want to proceed"))
            return(FALSE);
        initsec0(&MthrBlk);             /* initialize sector 0 for new disk */
        MthrBlk.date[0] = 0xff;         /* set partial format */
                                        /* write sector 0 unconditionally */
        if (capacity.blklen == BLKSIZE) {
            xprintf("Writing sector 0\n");
            if (retcode = sc_writenc(&MthrBlk, 0)) {
                DBG(printf(">After PWRITS0:  $%x", retcode);            )
                DBG(printf(" - %s\n", sc_derrmsg(errno));        )
                printf("Cannot write sector 0.");
		if((retcode>>8)&0xff == MEDIUM_ERR)
			goto try_formatting;
		else
                	goto format_aborted;
            }
                                            /* write sector 1 unconditionally */
            xprintf("Writing sector 1\n");
            if (retcode = sc_writenc(&MthrBlk, 1)) {
                DBG(printf(">After PWRITS1:  $%x", retcode);            )
                DBG(printf(" - %s\n", sc_derrmsg(errno));        )
                printf("Cannot write sector 1.\n");
                goto format_aborted;
            }
        }
        else {
            xprintf("Disk block size is not %d.\n", BLKSIZE);
        }

try_formatting:
	if (scsi_drv[scsi_drvtype].s_select) { /* is mode select supported? */

	        xprintf("Writing error recovery page\n");
	        merror.datalen = 0;             /* blame micropolis drives! */
	        merror.medtype = 0;
	        merror.wp = 0;
	        merror.bdlen = 8;
	        merror.density = 0;
	        memset(merror.numlb, 0, 3);
	        itoa3(merror.lblen, BLKSIZE);
	        merror.errec = 0;
	        if (scsi_drv[scsi_drvtype].s_autospare) {
	            merror.errec |= SCSI_M_ER_AWRE; /* enable auto write alloc */
                    merror.errec |= SCSI_M_ER_ARRE; /* enable auto read alloc */
     		   }
	        merror.retry = SCSI_V_MAXRETRY; /* set number of internal retries */
	        merror.corrspan = 0;
	        merror.headoff = 0;
	        merror.strobe = 0;
	        merror.tlimit = 0;
	        retcode = sc_modeselect(SCSI_V_MS_ERROR, &merror);
	        if (retcode) {
	            DBG(printf(">After MOD SEL:  $%x", retcode);            )
	            printf("\n%s\n", sc_derrmsg(errno));
	            printf("Unable to write error recovery page.");
	            goto format_aborted;
       		 }

	        xprintf("Writing rigid disk parameters page\n");
	        mgeo.datalen = 0;               /* blame micropolis drives */
	        mgeo.medtype = 0;
	        mgeo.wp = 0;
	        mgeo.bdlen = 8;
	        mgeo.density = 0;
	        memset(mgeo.numlb, 0, 3);
	        itoa3(mgeo.lblen, BLKSIZE);
	        retcode = sc_modeselect(SCSI_V_MS_GEO, &mgeo);
	        if (retcode) {
	            DBG(printf(">After MOD SEL:  $%x", retcode);            )
	            printf("\n%s\n", sc_derrmsg(errno));
	            printf("Unable to write rigid disk parameters page.");
	            goto format_aborted;
	        }
	}

        if (scsi_drv[scsi_drvtype].fmtfunc) {
            xprintf("Calling additional drive specific function.\n");
            if (!(*scsi_drv[scsi_drvtype].fmtfunc)())
                goto format_aborted;
        }

	        xprintf("Writing format parameters page\n");
	        mformat.datalen = 0;            /* micropolis drive requires it!! */
	        mformat.medtype = 0;
	        mformat.wp = 0;
	        mformat.bdlen = 8;
	        mformat.density = 0;
	        memset(mformat.numlb, 0, 3);
	        itoa3(mformat.lblen, BLKSIZE);
	        mformat.tpz = 1;                /* sectors/track format */
	        mformat.aspz = 1;               /* alternate sectors per track */
	        mformat.atpz = 0;
	        mformat.atpv = 3;
	        mformat.spt = 0;                /* let drive figure this out */
	        mformat.secsize = BLKSIZE;      /* sector size in bytes */
	        mformat.inter = 1;              /* interleave factor */
	        mformat.tskew = 2;
	        mformat.cskew = 0;
	        retcode = sc_modeselect(SCSI_V_MS_FORMAT, &mformat);
	        if (retcode) {
	            DBG(printf(">After MOD SEL:  $%x", retcode);            )
	            printf("\n%s\n", sc_derrmsg(errno));
	            printf("Unable to write format parameters page.");
	            goto format_aborted;
	        }

        if (scsi_drv[scsi_drvtype].fmttime)
            printf("Formatting will take about %d minutes to complete.\n",
                    scsi_drv[scsi_drvtype].fmttime);
        printf("Formatting...");
        if (retcode = sc_format()) {    /* format! */
            DBG(printf(">After FORMAT:   $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to complete format procedure.");
            goto format_aborted;
        }
        printf("Done!\n");

        if (!rewrite_sec01())           /* we do this again to get correct values */
            goto format_aborted;

        return(TRUE);                   /* successful! */

format_aborted:
        printf("  Format aborted!\n");
        return(FALSE);
}

rewrite_sec01()
{
        register int retcode;

        xprintf("Reading current error recovery page\n");
        retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_ERROR, &merror);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read error recovery page.");
            return(FALSE);
        }

        xprintf("Reading current format parameters page\n");
        retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_FORMAT, &mformat);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read format parameters page.");
            return(FALSE);
        }

        xprintf("Reading current rigid disk parameters page\n");
        retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_GEO, &mgeo);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read rigid disk parameters page.");
            return(FALSE);
        }

        xprintf("Reading drive capacity\n");
        if (retcode = sc_readcap(&capacity)) {
            DBG(printf(">After READ CP:  $%x", retcode);    )
            DBG(printf(" - %s\n", sc_derrmsg(errno));)
            printf("Cannot read capacity.");
            return(FALSE);
        }
        if (capacity.blklen != BLKSIZE) {
            printf("Block size is not 1024 but %d.", capacity.blklen);
            return(FALSE);
        }

        initsec0(&MthrBlk);             /* initialize with new data */
                                        /* write sector 0 */
        xprintf("Writing sector 0 (PWRITS0)\n");
        if (retcode = sc_writenc(&MthrBlk, 0)) {
            DBG(printf(">After PWRITS0:  $%x", retcode);            )
            DBG(printf(" - %s\n", sc_derrmsg(retcode));        )
            printf("Cannot write sector 0.");
            return(FALSE);
        }
                                        /* write sector 1 */
        xprintf("Writing sector 1 (PWRITS0)\n");
        if (retcode = sc_writenc(&MthrBlk, 1)) {
            DBG(printf(">After PWRITS0:  $%x", retcode);            )
            DBG(printf(" - %s\n", sc_derrmsg(errno));        )
            printf("Cannot write sector 1.");
            return(FALSE);
        }

        xprintf("Initializing the boot sector 8.\n");
        memset(CompBuf, 0, BLKSIZE);
        if (retcode = sc_writenc(CompBuf, 8)) {
            DBG(printf(">After PWRITS8:  $%x", retcode);            )
            DBG(printf(" - %s\n", sc_derrmsg(errno));        )
            printf("Cannot write sector 8.");
            return(FALSE);
        }
        return(TRUE);
}

/*---------------------------------------------------- initsec0() ------------
/ This function assumes that "mgeo", "mformat", and "capacity" are filled with
/ the correct values
/---------------------------------------------------------------------------*/
static int initsec0(sp)
struct sector0 *sp;
{
        memset(sp, 0, sizeof (struct sector0));
        strcpy(sp->id, "INIT");         /* signature */
        sp->date[0] = version;          /* major version number */
        sp->date[1] = release;          /* minor version number */
        strcpy(&sp->date[2], "880625"); /* date of somekind, I guess */
        sp->pd_ldmaxnum = LOGDR;        /* max # of logical drives */
        sp->pd_ldnum = 0;               /* logical drives defined */
        sp->nextsec = 0x12;             /* next available sector */
        sp->cyldisk = a3toi(mgeo.cyl);  /* get # of cylinders */
        sp->headcyl = mgeo.heads;       /* # of heads */
        sp->sechead = mformat.spt;      /* sectors/head */
        sp->seccyl = sp->sechead * sp->headcyl;
        sp->bytesec = BLKSIZE;          /* sector size */
	sp->rec_interlace = scsi_drv[scsi_drvtype].gap;
	sp->synchronous_drive = scsi_drv[scsi_drvtype].s_sync;
        sp->isscsi = 1;                 /* this is SCSI disk */
        sp->secdisk = capacity.lbaddr;  /* actual blocks available */
        sp->rv_strt = 0;                /* reserved area starting sector */
        sp->rv_size = 0x280;            /* reserved area size in sectors */
        if (sp->sechead)                /* round to track */
            sp->rv_size += sp->sechead - sp->rv_size % sp->sechead;
}

/*---------------------------------------------------- fmt_cdc() -------------
/ special format function for CDC WREN IV Model 94171 drives only
/---------------------------------------------------------------------------*/
fmt_cdc()
{
        struct _scsi_mode_cache mcache; /* MODE SELECT/SENSE */
        register int retcode;

        xprintf("Reading cache control parameters page\n");
        retcode = sc_modesense(SCSI_V_PC_SAV, SCSI_V_MS_CACHE, &mcache);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read cache control parameters page.");
            return(FALSE);
        }

        mcache.datalen = 0;
        mcache.medtype = 0;
        mcache.wp = 0;
        mcache.bdlen = 8;
        mcache.density = 0;
        memset(mcache.numlb, 0, sizeof (mcache.numlb));
        itoa3(mcache.lblen, BLKSIZE);
        mcache.pcache |= SCSI_M_CC_CE;  /* enable cache */
        xprintf("Writing cache control parameters page\n");
        retcode = sc_modeselect(SCSI_V_MS_CACHE, &mcache);
        if (retcode) {
            DBG(printf(">After MOD SEL:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to write cache control parameters page.");
            return(FALSE);
        }
        return(TRUE);
}

/*----------------------------- End of scsiinit.c --------------------------*/
