/*----------------------------------------------------------------------------
/ scsifunc.c - scsi device support functions
/
/ Craig J. Kim      June 1988
/ (c) Copyright 1988 ARIX Corp.  San Jose, CA
/
/ All SCSI functions are called through devreq() CPU PROM which hands the
/ request down to the appropriate SCSI controller.  The code on SCSI
/ controller attempts the request and sends back the result such that when
/ an error occurs, it calls REQUEST SENSE automatically and determines the
/ actual cause of the error and passes back the result to the caller.
/ Therefore, it is NOT necessary for SCSI funtions to call REQUEST SENSE
/ again to determine the actual cause of the error.  If REQUEST SENSE is
/ called again, it will return no error condition.
/---------------------------------------------------------------------------*/

#include "scsidt.h"

#define SCSIBUFSIZ          4096        /* write upto 4 blocks at once */

/*---------------------------------------------------- forwards ------------*/
int sc_finitialize();                   /* scsi function variable init */
char *sc_errmsg();                      /* scsi error message */
int sc_format();                        /* format */
int sc_inquiry();                       /* inquiry */
int sc_modeselect();                    /* mode select */
int sc_modesense();                     /* mode sense */
int sc_read();                          /* read */
int sc_readcap();                       /* read capacity */
int sc_readxt();                        /* read extended */
int sc_xreqsense();                     /* request sense - extended */
int sc_rezero();                        /* rezero */
int sc_seek();                          /* seek */
int sc_seekxt();                        /* seek extended */
int sc_start();                         /* start/stop */
int sc_testunit();                      /* test unit ready */
int sc_write();                         /* write */
int sc_writext();                       /* write extended */

/*---------------------------------------------------- globals -------------*/
static char reqmem[SCSIBUFSIZ + 4];     /* request input/output buffer */
static char *reqmemp = reqmem;          /* points reqmem long word aligned */

/*---------------------------------------------------- sc_finitialize() ------
/ call this routine before any other scsi functions
/---------------------------------------------------------------------------*/
int sc_finitialize()
{
        /* must be long word aligned */
        reqmemp = (char *) ((unsigned) reqmem & ~0x03);
}

/*---------------------------------------------------- sc_errmsg() -----------
/ returns a pointer to a message for the passed error code
/---------------------------------------------------------------------------*/
char *sc_errmsg(errcode)
register int errcode;
{
        static char *scsi_errmsg[] = {
            "",
            "Recovered error",
            "Not ready",
            "Medium error",
            "Hardware error",
            "Illegal request",
            "Unit attention",
            "Data protect",
            "Reserved",
            "Vendor unique",
            "Copy/compare aborted",
            "Aborted command",
            "Reserved",
            "Reserved",
            "Miscompare",
            "Reserved"              };

        if (errcode < 0 || errcode > sizeof (scsi_errmsg) / sizeof (char *))
            return("Unknown");
        return(scsi_errmsg[errcode]);
}

/*---------------------------------------------------- sc_derrmsg() ----------
/ returns a pointer to a message for the passed error code
/---------------------------------------------------------------------------*/
char *sc_derrmsg(errcode)
register int errcode;
{
        static char *devdrv_errmsg[] = {
            "",
            "Can't write to system sector",
            "Illegal disk command",     /* 2 */
            "Invalid sector num",
            "Disk not formatted",
            "Invalid system sector information in sector 0-17",
            "Drive not ready",          /* 6 */
            "Device check error",
            "No such logical drive",
            "Physical drive out of range",
            "Devtype is not disk or tape",
            "Illegal byte count in request",
            "Skip track table is full",
            "Invalid skip track list",
            "Hit alternate sector with id field = -1",
            "Too many sectors",
            "Hit alternate sector",     /* hard errors */
            "Data CRC error",
            "Error during SEEK",
            "Ready change",
            "Can't rezero during retry",
            "Not on cylinder",
            "Header ID error",
            "Reserved",
            "Overrun",
            "Reserved",
            "Write protected",
            "Memory address not a long word multiple",
            "Header search",
            "Timeout on rw",
            "Incorrect timing values for r/w a sector or r/w a sector id (sector 0 should have the correct timing parameters)",
            "Abnormal termination - hsdt board disk interface is hung",
            "RECOVERED_ERR",            /* SCSI sense keys */
            "MEDIUM_ERR",
            "HARDWARE_ERR",
            "ILLEGAL_REQ",
            "ABORTED_CMD",
            "VOLUME_OVRFLOW",
            "MISCMPR",
            "Invalid length in a request",
            "A request past the end of physical media on the disk",
            "Failure of one of the two scsi controller channels",
            "Reserved",
            "Reserved",
            "Reserved",
            "Reserved",
            "Reserved",
            "Reserved",
            "Cartridge not in place",   /* tape errors */
            "Tape drive not ready",
            "Write protected",
            "End of media",
            "Unrecoverable data err  or BIE(block in error) not located",
            "File mark detected",
            "No data detected",
            "8 or more retries",
            "Beginning of media",
            "Tape drive is hung",
            "Tape drive was reset",
            "Tape not online",
            "Data overrun, tapeblk is bigger than request",
            "9 track, error during write operation",
            "Illegal tape cmd",
            "Invalid io interface",
            "Tape request abort,due to ups",
            "No nine track tape on this controller",
            "No burst id detected",
            "Invalid density selection",
            "Invalid tape speed selection",
            "Invalid tape mode selection",
            "Tape drive is in the process of performing a read/write request",
            "Invalid main memory address",
            "Invalid byte count in request"     };

        if (errcode < 0 || errcode > sizeof (devdrv_errmsg) / sizeof (char *))
            return("Unknown");
        return(devdrv_errmsg[errcode]);
}

/*---------------------------------------------------- sc_format() -----------
/ FORMAT command
/ By default, scsi driver will use the interleave factor of 1.  UNIX will use
/ its own software interleave factor.
/---------------------------------------------------------------------------*/
int sc_format()
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.opr = SCSI_V_CMD_FORMAT;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_inquiry() ----------
/ INQUIRY command
/ The command provides a means by which the initiator may request information
/ regarding the drive.
/---------------------------------------------------------------------------*/
int sc_inquiry(inq)
struct _scsi_inquiry *inq;
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_mem = reqmemp;
        sc_devreq.q_count = sizeof (struct _scsi_inquiry);
        sc_devreq.opr = SCSI_V_CMD_INQUIRY;
        devreq(SlotPtr, &sc_devreq);

        memcpy((char *) inq, reqmemp, sizeof (struct _scsi_inquiry));
        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_modeselect() -------
/ MODE SELECT command
/ The command provides a means by which the initiator may specify logical
/ unit and/or periphral device parameters to the drive.
/---------------------------------------------------------------------------*/
int sc_modeselect(page, mem)
int page;               /* page to request */
char *mem;              /* where to get page info from */
{
        register int q_count;
        struct _scsi_mode_sense  *mp;
        struct _scsi_mode_error  *ep;
        struct _scsi_mode_conn   *cp;
        struct _scsi_mode_format *fp;
        struct _scsi_mode_geo    *gp;
        struct _scsi_mode_flex   *xp;
        struct _scsi_mode_cache  *ap;

        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_mem = reqmemp;
        switch (page) {
            case SCSI_V_MS_MS:          /* Mode sense */
                mp = (struct _scsi_mode_sense *) mem;
                mp->bdlen = 8;
                q_count = sizeof (struct _scsi_mode_sense);
                break;
            case SCSI_V_MS_ERROR:       /* error recovery parameters */
                ep = (struct _scsi_mode_error *) mem;
                ep->medtype = 0;
                ep->bdlen = 8;
                ep->pagecode = SCSI_V_MS_ERROR;
                ep->pagelen = SCSI_V_MS_ERRLEN;
                q_count = sizeof (struct _scsi_mode_error);
                break;
            case SCSI_V_MS_CONN:        /* disconnect/reconnect control parm */
                cp = (struct _scsi_mode_conn *) mem;
                cp->medtype = 0;
                cp->bdlen = 8;
                cp->pagecode = SCSI_V_MS_CONN;
                cp->pagelen = SCSI_V_MS_CONLEN;
                q_count = sizeof (struct _scsi_mode_conn);
                break;
            case SCSI_V_MS_FORMAT:      /* direct access device format parms */
                fp = (struct _scsi_mode_format *) mem;
                fp->medtype = 0;
                fp->bdlen = 8;
                fp->pagecode = SCSI_V_MS_FORMAT;
                fp->pagelen = SCSI_V_MS_FORLEN;
                q_count = sizeof (struct _scsi_mode_format);
                break;
            case SCSI_V_MS_GEO:         /* rigid disk geometry parms */
                gp = (struct _scsi_mode_geo *) mem;
                gp->medtype = 0;
                gp->bdlen = 8;
                gp->pagecode = SCSI_V_MS_GEO;
                gp->pagelen = SCSI_V_MS_GEOLEN;
                q_count = sizeof (struct _scsi_mode_geo);
                break;
            case SCSI_V_MS_FLEX:        /* flexible disk drive parms */
                xp = (struct _scsi_mode_flex *) mem;
                xp->medtype = 0;
                xp->bdlen = 8;
                xp->pagecode = SCSI_V_MS_FLEX;
                xp->pagelen = SCSI_V_MS_FLXLEN;
                q_count = sizeof (struct _scsi_mode_flex);
                break;
            case SCSI_V_MS_CACHE:       /* cache control */
                ap = (struct _scsi_mode_cache *) mem;
                ap->medtype = 0;
                ap->bdlen = 8;
                ap->pagecode = SCSI_V_MS_CACHE;
                ap->pagelen = SCSI_V_MS_CACLEN;
                q_count = sizeof (struct _scsi_mode_cache);
                break;
            default:
                printf("Unknown SCSI MODE page for MODE SELECT.\n");
                return(2);
                break;
        }
        memcpy(reqmemp, mem, q_count);
        sc_devreq.q_count = q_count;
        sc_devreq.opr = SCSI_V_CMD_MODE_SELECT;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_modesense() --------
/ MODE SENSE command
/ The command provides a means by which the initiator may receive the logical
/ unit and periphral device parameters form the drive.
/
/ The returned structures may not contain valid data unless the error codes
/ are valid.
/---------------------------------------------------------------------------*/
int sc_modesense(pcf, page, mem)
unsigned int pcf;       /* page control */
unsigned int page;      /* page to request */
char *mem;              /* where to put page info */
{
        register int q_count;
        struct _scsi_mode_sense  *mp;
        struct _scsi_mode_error  *ep;
        struct _scsi_mode_conn   *cp;
        struct _scsi_mode_format *fp;
        struct _scsi_mode_geo    *gp;
        struct _scsi_mode_flex   *xp;
        struct _scsi_mode_cache  *ap;

        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        page &= SCSI_M_MS_PAGE;
        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_mem = reqmemp;
        memset(reqmemp, 0, 64);
        switch (page) {
            case SCSI_V_MS_MS:          /* Mode sense */
                pcf = 0;
                mp = (struct _scsi_mode_sense *) reqmemp;
                mp->datalen = (uchar) (3 + 8);
                mp->bdlen = (uchar) 8;
                q_count = sizeof (struct _scsi_mode_sense);
                break;
            case SCSI_V_MS_ERROR:       /* error recovery parameters */
                ep = (struct _scsi_mode_error *) reqmemp;
                ep->datalen = (uchar) (3 + 8 + 2 + SCSI_V_MS_ERRLEN);
                ep->bdlen = (uchar) 8;
                ep->pagecode = (uchar) ((pcf << SCSI_S_MS_PCF) | SCSI_V_MS_ERROR);
                ep->pagelen = (uchar) SCSI_V_MS_ERRLEN;
                q_count = sizeof (struct _scsi_mode_error);
                break;
            case SCSI_V_MS_CONN:        /* disconnect/reconnect control parm */
                cp = (struct _scsi_mode_conn *) reqmemp;
                cp->datalen = (uchar) (3 + 8 + 2 + SCSI_V_MS_CONLEN);
                cp->bdlen = (uchar) 8;
                cp->pagecode = (uchar) ((pcf << SCSI_S_MS_PCF) | SCSI_V_MS_CONN);
                cp->pagelen = (uchar) SCSI_V_MS_CONLEN;
                q_count = sizeof (struct _scsi_mode_conn);
                break;
            case SCSI_V_MS_FORMAT:      /* direct access device format parms */
                fp = (struct _scsi_mode_format *) reqmemp;
                fp->datalen = (uchar) (3 + 8 + 2 + SCSI_V_MS_FORLEN);
                fp->bdlen = (uchar) 8;
                fp->pagecode = (uchar) ((pcf << SCSI_S_MS_PCF) | SCSI_V_MS_FORMAT);
                fp->pagelen = (uchar) SCSI_V_MS_FORLEN;
                q_count = sizeof (struct _scsi_mode_format);
                break;
            case SCSI_V_MS_GEO:         /* rigid disk geometry parms */
                gp = (struct _scsi_mode_geo *) reqmemp;
                gp->datalen = (uchar) (3 + 8 + 2 + SCSI_V_MS_GEOLEN);
                gp->bdlen = (uchar) 8;
                gp->pagecode = (uchar) ((pcf << SCSI_S_MS_PCF) | SCSI_V_MS_GEO);
                gp->pagelen = (uchar) SCSI_V_MS_GEOLEN;
                q_count = sizeof (struct _scsi_mode_geo);
                break;
            case SCSI_V_MS_FLEX:        /* flexible disk drive parms */
                xp = (struct _scsi_mode_flex *) reqmemp;
                xp->datalen = (uchar) (3 + 8 + 2 + SCSI_V_MS_FLXLEN);
                xp->bdlen = (uchar) 8;
                xp->pagecode = (uchar) ((pcf << SCSI_S_MS_PCF) | SCSI_V_MS_FLEX);
                xp->pagelen = (uchar) SCSI_V_MS_FLXLEN;
                q_count = sizeof (struct _scsi_mode_flex);
                break;
            case SCSI_V_MS_CACHE:       /* cache control */
                ap = (struct _scsi_mode_cache *) reqmemp;
                ap->datalen = (uchar) (3 + 8 + 2 + SCSI_V_MS_CACLEN);
                ap->bdlen = (uchar) 8;
                ap->pagecode = (uchar) ((pcf << SCSI_S_MS_PCF) | SCSI_V_MS_CACHE);
                ap->pagelen = (uchar) SCSI_V_MS_CACLEN;
                q_count = sizeof (struct _scsi_mode_cache);
                break;
            default:
                printf("Unknown SCSI MODE page for MODE SENSE.\n");
                return(2);
                break;
        }
        sc_devreq.q_count = q_count;
        sc_devreq.badsecn = (uchar) ((pcf << SCSI_S_MS_PCF) | page);
/*        sc_devreq.scsi_page = (uchar) ((pcf << SCSI_S_MS_PCF) | page); */
        sc_devreq.opr = SCSI_V_CMD_MODE_SENSE;
        devreq(SlotPtr, &sc_devreq);

        memcpy(mem, reqmemp, q_count);
        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_read() -------------
/ READ command is implemented using normal device driver call rather than
/ scsi special call.
/ WARNING:  The passed memory location must be at even long word boundary for
/ this function to work correctly.  The function assumes that there are enough
/ memory to store passed number of disk blocks.
/---------------------------------------------------------------------------*/
int sc_read(dest, blkstrt, numblks)
char *dest;             /* block content */
block_t blkstrt;        /* starting block number */
int numblks;            /* number of blocks to read */
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = PREADPS;      /* not a scsi call */
        sc_devreq.q_devun.block = blkstrt;
        sc_devreq.q_mem = dest;
        sc_devreq.q_count = numblks * BLKSIZE;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_readcap() ----------
/ READ CAPACITY command
/---------------------------------------------------------------------------*/
int sc_readcap(cap)
struct _scsi_read_cap *cap;
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_mem = reqmemp;
        sc_devreq.q_count = sizeof (struct _scsi_read_cap);
        sc_devreq.opr = SCSI_V_CMD_READ_CAPACITY;
        devreq(SlotPtr, &sc_devreq);

        memcpy((char *) cap, reqmemp, sizeof (struct _scsi_read_cap));
        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_readxt() -----------
/ READ EXTENDED command
/ passed buffer must be long word aligned for this function to work.
/---------------------------------------------------------------------------*/
int sc_readxt(dest, blkstrt, numblks)
char *dest;             /* block content */
block_t blkstrt;        /* starting block number */
int numblks;            /* number of blocks to read */
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_devun.block = blkstrt;
        sc_devreq.q_mem = dest;
        sc_devreq.q_count = numblks * BLKSIZE;
        sc_devreq.opr = SCSI_V_CMD_READ_EXTENDED;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_reassign() ---------
/ REASSIGN BLOCKS command
/---------------------------------------------------------------------------*/
int sc_reassign(numblks, blocks)
int numblks;
uint blocks[];
{
        struct _scsi_reassign *rs;
        int bytecount = 4 + 4 * numblks;

        if (bytecount >= SCSIBUFSIZ) {  /* check for limitation */
            errno = 0x1d;               /* too many */
            return(0x1d00);
        }
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        rs = (struct _scsi_reassign *) reqmemp;
        rs->zero = 0;                   /* this must be zero */
        rs->listlen = (ushort) numblks; /* number of blocks */
        memcpy(rs->block, blocks, 4 * numblks);
        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_mem = reqmemp;
        sc_devreq.q_count = bytecount;
        sc_devreq.opr = SCSI_V_CMD_REASSIGN;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_xreqsense() --------
/ REQUEST SENSE command - extended
/---------------------------------------------------------------------------*/
int sc_xreqsense(xreq)
struct _scsi_xreq_sense *xreq;
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_mem = reqmemp;
        sc_devreq.q_count = sizeof (struct _scsi_xreq_sense);
        sc_devreq.opr = SCSI_V_CMD_REQ_SENSE;
        devreq(SlotPtr, &sc_devreq);

        memcpy((char *) xreq, reqmemp, sizeof (struct _scsi_xreq_sense));
        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_rezero() -----------
/ REZERO command
/ The command requests that the drive set the logical unit to logical block
/ address zero.
/---------------------------------------------------------------------------*/
int sc_rezero()
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.opr = SCSI_V_CMD_REZERO;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_seek() -------------
/ SEEK command
/---------------------------------------------------------------------------*/
sc_seek(block)
block_t block;          /* seeking block number */
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_devun.block = block;
        sc_devreq.opr = SCSI_V_CMD_SEEK;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_seekxt() -----------
/ SEEK EXTENDED command
/---------------------------------------------------------------------------*/
int sc_seekxt(block)
block_t block;          /* seeking block number */
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_devun.block = block;
        sc_devreq.opr = SCSI_V_CMD_SEEK_EXTENDED;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_start() ------------
/ START/STOP command
/---------------------------------------------------------------------------*/
sc_start()
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.opr = SCSI_V_CMD_START_UNIT;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_testunit() ---------
/ TEST UNIT READY command
/---------------------------------------------------------------------------*/
int sc_testunit()
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.opr = SCSI_V_CMD_UNIT_READY;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_write() ------------
/ WRITE command is implemented using normal device driver call instead of
/ scsi special calls.
/ The command requests that the drive write the data transferred by the
/ initiator to the logical unit.
/ WARNING:  The passed memory location must be at even long word boundary for
/ this function to work correctly.
/---------------------------------------------------------------------------*/
int sc_write(source, blkstrt, numblks)
char *source;           /* block content to be written */
block_t blkstrt;        /* starting block number */
int numblks;            /* number of blocks to read */
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = PWRITPS;      /* not a scsi call */
        sc_devreq.q_devun.block = blkstrt;
        sc_devreq.q_mem = source;
        sc_devreq.q_count = numblks * BLKSIZE;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_writenc() ----------
/ This function is a special write to disk without checking anything.  This
/ is useful to write sector 0 for unformatted disks and should NOT be used
/ for anything else!
/ The command requests that the drive write the data transferred by the
/ initiator to the logical unit.
/ WARNING:  The passed memory location must be at even long word boundary for
/ this function to work correctly.
/---------------------------------------------------------------------------*/
int sc_writenc(block, blknum)
char *block;            /* block content to be written */
block_t blknum;         /* starting block number */
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = PWRITS0;      /* not a scsi call */
        sc_devreq.q_devun.block = blknum;
        sc_devreq.q_mem = block;
        sc_devreq.q_count = BLKSIZE;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*---------------------------------------------------- sc_writext() ----------
/ WRITE EXTENDED command
/ The command requests that the drive write the data transferred by the
/ initiator to the logical unit.
/ Passed buffer must be long word aligned for this function to work.
/---------------------------------------------------------------------------*/
int sc_writext(source, blkstrt, numblks)
char *source;           /* block content to be written */
block_t blkstrt;        /* starting block number */
int numblks;            /* number of blocks to read */
{
        memset((char *) &sc_devreq, 0, sizeof (sc_devreq));

        sc_devreq.q_devtype = DTDISK;
        sc_devreq.q_devnum = DriveNum;
        sc_devreq.q_cmd = SCSI_CMD;
        sc_devreq.q_devun.block = blkstrt;
        sc_devreq.q_mem = source;
        sc_devreq.q_count = numblks * BLKSIZE;
        sc_devreq.opr = SCSI_V_CMD_WRITE_EXTENDED;
        devreq(SlotPtr, &sc_devreq);

        errno = sc_devreq.rc1;
        return(*((unsigned short *) &sc_devreq.rc1));
}

/*----------------------------- End of scsifunc.c --------------------------*/
