/*----------------------------------------------------------------------------
/ scsi.h - SCSI command structures
/
/ Craig J. Kim      June 1988
/ (c) Copyright 1988 ARIX Corp.
/
/ Convention:
/   All structure names starts with '_scsi' to avoid confusion.  All macros
/   has format:
/
/       SCSI_X_XY_XYZ
/       ---- - -- ----
/         |  |  |   |
/         |  |  |   `-------------- description of macro
/         |  |  `------------------ class - two character code
/         |  `--------------------- type - one character
/         |                           E = error code
/         |                           M = bit mask
/         |                           S = bit shifting
/         |                           T = type
/         |                           V = value
/         `------------------------ must be SCSI
/
/---------------------------------------------------------------------------*/

#define SCSI_CMD        0x25            /* scsi command code to dev drvr */

/* typedef unsigned char   uchar;          /* one byte */
/* typedef unsigned short  ushort;         /* two byte */
/* typedef unsigned long   ulong;          /* four byte */
typedef unsigned long   block_t;        /* block numbers */

#define DEVERR(x)    (x & 0xff)         /* device error code */
#define DDERR(x)   ((x & 0xff00) >> 8)  /* device driver error code */

#ifndef TRUE
#define TRUE            1
#define FALSE           0
#endif

#define SCSI_NUMDRVS    5               /* number of drives supported */

struct _scsi_drvparms {
        char *diskname;                 /* drive id codes */
        uchar namelen;                  /* length to compare */
        ushort cyldisk;                 /* # of cyls per disk */
        uchar headcyl;                  /* # of heads per cylinder */
        uchar sechead;                  /* # of sectors per track */
        ushort bytesec;                 /* # of data bytes per sector */
        ushort trbps;                   /* # of bits + gap per sector */
        ushort ltrbps;                  /* # of bits + gap last sector */
        char autosz;                    /* ok to poke non-existent heads */
        char gap;                       /* recommended disk interlace gap */
        int (*fmtfunc)();               /* format function */
        int (*initfunc)();              /* initialization function */
};

/*----------------------------- INQUIRY --------------------------------------
/ This command returns the device information
/---------------------------------------------------------------------------*/
struct _scsi_inquiry {
        uchar devtype;                  /* peripheral device type */
#define   SCSI_T_DT_DIR         0x00    /* direct access device (disk) */
#define   SCSI_T_DT_SEQ         0x01    /* sequential access device (tape) */
#define   SCSI_T_DT_PRT         0x02    /* printer device */
#define   SCSI_T_DT_PRC         0x03    /* processor device */
#define   SCSI_T_DT_WORM        0x04    /* write-once read-multiple */
#define   SCSI_T_DT_RO          0x05    /* read-only direct-access */
#define   SCSI_T_DT_NOLU        0x07    /* logical unit not present */
        uchar devqual;                  /* device type qualifier */
#define   SCSI_M_DQ_RMD         0x80    /* removal medium */
#define   SCSI_M_DQ_DTQ         0x7f    /* device type qualifier */
        uchar version;                  /* version information */
#define   SCSI_M_VR_ISO         0xc0    /* ISO version */
#define   SCSI_M_VR_ECMA        0x38    /* ECMA version */
#define   SCSI_M_VR_ANSI        0x07    /* ANSI version */
#define     SCSI_V_VA_NON       0x00    /* no version specified */
#define     SCSI_V_VA_NOW       0x01    /* current version */
        uchar reserved1;                /* reserved */
        uchar addlen;                   /* additional length */
        uchar reserved2;                /* reserved */
        uchar reserved3;                /* reserved */
        uchar reserved4;                /* reserved */
        char idvendor[8];               /* ascii vendor id name */
        char idprod[16];                /* ascii product id */
        char idrev[4];                  /* ascii revision id */
};

/*----------------------------- MODE SENSE -----------------------------------
/ The MODE SENSE command provides a means for a target to report its medium,
/ logical unit, or periphral device parameters to the initiator.
/---------------------------------------------------------------------------*/
#define SCSI_M_MS_PCF           0xc0    /* page control field */
#define SCSI_S_MS_PCF           0x06    /* shift six bits for bits 6 & 7 */
#define SCSI_V_PC_CUR           0x00    /* report current value */
#define SCSI_V_PC_CHNG          0x01    /* report changeable values */
#define SCSI_V_PC_DEF           0x02    /* report default values */
#define SCSI_V_PC_SAV           0x03    /* report saved values */

#define SCSI_M_MS_PAGE          0x3f    /* page code */
#define SCSI_V_MS_MS            0x00    /* Mode sense */
#define SCSI_V_MS_ERROR         0x01    /* error recovery parameters */
#define SCSI_V_MS_CONN          0x02    /* disconnect/reconnect control parm */
#define SCSI_V_MS_FORMAT        0x03    /* direct access device format parms */
#define SCSI_V_MS_GEO           0x04    /* rigid disk geometry parms */
#define SCSI_V_MS_FLEX          0x05    /* flexible disk drive parms */
#define SCSI_V_MS_CACHE         0x38    /* cache control */
#define SCSI_V_MS_ALL           0x3f    /* report all pages */

struct _scsi_mode_sense {               /* mode sense parameter list header */
        uchar datalen;                  /* sense data length */
        uchar medtype;                  /* medium type */
#define   SCSI_T_MT_DF          0x00    /* default - only one density */
#define   SCSI_T_MT_SD          0x01    /* flexible disk, single density */
#define   SCSI_T_MT_DD          0x02    /* flexible disk, double density */
        uchar wp;                       /* write protect / reserved */
#define   SCSI_M_WP_WP          0x80    /* write protect */
        uchar bdlen;                    /* block descriptor length */
        uchar density;                  /* density */
        uchar numlb[3];                 /* number of logical blocks */
        uchar reserved;                 /* reserved */
        uchar lblen[3];                 /* logical block length */
};

struct _scsi_mode_error {               /* mode sense error page descriptor */
        uchar datalen;                  /* sense data length */
        uchar medtype;                  /* medium type */
        uchar wp;                       /* write protect / reserved */
        uchar bdlen;                    /* block descriptor length */
        uchar density;                  /* density */
        uchar numlb[3];                 /* number of logical blocks */
        uchar reserved;                 /* reserved */
        uchar lblen[3];                 /* logical block length */
        uchar pagecode;                 /* page code for this descriptor */
#define   SCSI_M_MS_PS          0x80    /* page savable */
#define   SCSI_M_MS_PAGECODE    0x3f    /* page code mask */
        uchar pagelen;                  /* page length in bytes */
#define   SCSI_V_MS_ERRLEN      0x06    /* length of page */
        uchar errec;                    /* recovery parameters */
#define   SCSI_M_ER_AWRE        0x80    /* auto write realloc enabled */
#define   SCSI_M_ER_ARRE        0x40    /* auto read realloc enabled */
#define   SCSI_M_ER_TB          0x20    /* transfer block */
#define   SCSI_M_ER_RC          0x10    /* read continuous */
#define   SCSI_M_ER_EEC         0x08    /* enable early correction */
#define   SCSI_M_ER_PER         0x04    /* post error */
#define   SCSI_M_ER_DTE         0x02    /* disable transfer on error */
#define   SCSI_M_ER_DCR         0x01    /* disable correction */
        uchar retry;                    /* retry counts */
#define   SCSI_V_MAXRETRY 0x04          /* try this many times */
        uchar corrspan;                 /* correction span */
        uchar headoff;                  /* head offset count */
        uchar strobe;                   /* data strobe offset count */
        uchar tlimit;                   /* recovery time limit */
};

struct _scsi_mode_conn {                /* mode sense disconnect page desc */
        uchar datalen;                  /* sense data length */
        uchar medtype;                  /* medium type */
        uchar wp;                       /* write protect / reserved */
        uchar bdlen;                    /* block descriptor length */
        uchar density;                  /* density */
        uchar numlb[3];                 /* number of logical blocks */
        uchar reserved1;                /* reserved */
        uchar lblen[3];                 /* logical block length */
        uchar pagecode;                 /* page code for this descriptor */
        uchar pagelen;                  /* page length in bytes */
#define   SCSI_V_MS_CONLEN      0x0a
        uchar bufull;                   /* buffer full ratio */
        uchar bufmt;                    /* buffer empty ratio */
        ushort bilim;                   /* bus inactivity limit */
        ushort dtlim;                   /* disconnect time limit */
        ushort ctlim;                   /* connect time limit */
        ushort reserved2;               /* reserved */
};

struct _scsi_mode_format {              /* mode sense format page desc */
        uchar datalen;                  /* sense data length */
        uchar medtype;                  /* medium type */
        uchar wp;                       /* write protect / reserved */
        uchar bdlen;                    /* block descriptor length */
        uchar density;                  /* density */
        uchar numlb[3];                 /* number of logical blocks */
        uchar reserved1;                /* reserved */
        uchar lblen[3];                 /* logical block length */
        uchar pagecode;                 /* page code for this descriptor */
        uchar pagelen;                  /* page length in bytes */
#define   SCSI_V_MS_FORLEN      0x16
        ushort tpz;                     /* tracks per zone */
        ushort aspz;                    /* alternate sectors per zone */
        ushort atpz;                    /* alternate tracks per zone */
        ushort atpv;                    /* alternate tracks per volume */
        ushort spt;                     /* sectors per track */
        ushort secsize;                 /* bytes per physical sector */
        ushort inter;                   /* interleave value */
        ushort tskew;                   /* track skew */
        ushort cskew;                   /* cylinder skew */
        uchar drvtype;                  /* drive type field */
#define   SCSI_M_DT_SSEC        0x80    /* soft sector format */
#define   SCSI_M_DT_HSEC        0x40    /* hard sector format */
#define   SCSI_M_DT_RMB         0x20    /* removable medium */
#define   SCSI_M_DT_SURF        0x10    /* surface */
#define   SCSI_M_DT_INS         0x08    /* inhibit save */
        uchar reserved2[3];             /* reserved */
};

struct _scsi_mode_geo {                 /* mode sense geometry page desc */
        uchar datalen;                  /* sense data length */
        uchar medtype;                  /* medium type */
        uchar wp;                       /* write protect / reserved */
        uchar bdlen;                    /* block descriptor length */
        uchar density;                  /* density */
        uchar numlb[3];                 /* number of logical blocks */
        uchar reserved1;                /* reserved */
        uchar lblen[3];                 /* logical block length */
        uchar pagecode;                 /* page code for this descriptor */
        uchar pagelen;                  /* page length in bytes */
#define   SCSI_V_MS_GEOLEN      0x12
        uchar cyl[3];                   /* maximum number of cylinders */
        uchar heads;                    /* maximum number of heads */
        uchar scylwp[3];                /* starting cyl - write precomp */
        uchar scylrw[3];                /* starting cyl - reduced write */
        ushort dstep;                   /* drive step rate */
        uchar land[3];                  /* landing zone cylinder */
        uchar reserved2[3];             /* reserved */
};

struct _scsi_mode_flex {                /* mode sense flexible disks */
        uchar datalen;                  /* sense data length */
        uchar medtype;                  /* medium type */
        uchar wp;                       /* write protect / reserved */
        uchar bdlen;                    /* block descriptor length */
        uchar density;                  /* density */
        uchar numlb[3];                 /* number of logical blocks */
        uchar reserved;                 /* reserved */
        uchar lblen[3];                 /* logical block length */
        uchar pagecode;                 /* page code for this descriptor */
        uchar pagelen;                  /* page length in bytes */
#define   SCSI_V_MS_FLXLEN      0x16
        ushort trate;                   /* transfer rate */
#define   SCSI_V_TR_250K        0x00fa  /* 250 kbit/second */
#define   SCSI_V_TR_300K        0x012c  /* 300 kbit/second */
#define   SCSI_V_TR_500K        0x01f4  /* 500 kbit/second */
#define   SCSI_V_TR_1M          0x03e8  /* 1 megabit/second */
#define   SCSI_V_TR_2M          0x07d0  /* 2 megabit/second */
#define   SCSI_V_TR_5M          0x1388  /* 5 megabit/second */
        uchar heads;                    /* number of heads */
        uchar sectors;                  /* sectors per track */
        ushort secsize;                 /* data bytes per physical sector */
        ushort cyl;                     /* number of cylinders */
        ushort scylwp;                  /* starting cyl - write precomp */
        ushort scylrw;                  /* starting cyl - reduced write */
        ushort dstep;                   /* drive step rate */
        uchar dwidth;                   /* drive step pulse width */
        uchar hsettle;                  /* head settle delay */
        uchar mtron;                    /* motor on delay */
        uchar mtroff;                   /* motor off delay */
        uchar trdy;                     /* true ready */
#define   SCSI_M_FX_TRDY        0x80    /* true ready mask */
        uchar hload;                    /* head load delay */
        uchar ssec0;                    /* starting sector, side zero */
        uchar ssec1;                    /* starting sector, side one */
};

struct _scsi_mode_cache {               /* mode sense cache control */
        uchar datalen;                  /* sense data length */
        uchar medtype;                  /* medium type */
        uchar wp;                       /* write protect / reserved */
        uchar bdlen;                    /* block descriptor length */
        uchar density;                  /* density */
        uchar numlb[3];                 /* number of logical blocks */
        uchar reserved1;                /* reserved */
        uchar lblen[3];                 /* logical block length */
        uchar pagecode;                 /* page code for this descriptor */
        uchar pagelen;                  /* page length in bytes */
#define   SCSI_V_MS_CACLEN      0x0e    /* */
        uchar pcache;                   /* cache control parameters */
#define   SCSI_M_CC_WIE         0x40    /* write index enable */
#define   SCSI_M_CC_CE          0x10    /* cache enable */
#define   SCSI_M_CC_CTS         0x0f    /* cache table size */
        uchar thresh;                   /* prefetch thresh hold */
        uchar maxfetch;                 /* maximum prefetch */
        uchar maxpfm;                   /* maximum prefetch multiplier */
        uchar minfetch;                 /* minimum prefetch */
        uchar minpfm;                   /* minimum prefetch multiplier */
        uchar reserved2[8];             /* reserved */
 };

/*-------------------------------- READ --------------------------------------
/ Errors detected by the drive during the READ command will cause the drive
/ to terminate the READ command with a CHECK CONDITION status.  Error
/ information should be requested by the Initiator via the REQUEST SENSE
/ command.
/---------------------------------------------------------------------------*/

/*---------------------------- READ CAPACITY ---------------------------------
/ The READ CAPACITY command provides a means for the initiator to request
/ information regarding the capacity of the logical unit.
/---------------------------------------------------------------------------*/
struct _scsi_read_cap {                 /* read capacity data format */
        ulong lbaddr;                   /* logical block address */
        ulong blklen;                   /* block length in bytes */
};

/*--------------------------- REQUEST SENSE ----------------------------------
/ This command returns the error conditions of last command.
/---------------------------------------------------------------------------*/
struct _scsi_xreq_sense {
        uchar valid;                    /* valid field */
        uchar segment;                  /* segment number */
        uchar key;                      /* extended sense data format */
#define   SCSI_M_SK_FMARK       0x80    /* filemark (bit 7) */
#define   SCSI_M_SK_EOM         0x40    /* end of medium (bit 6) */
#define   SCSI_M_SK_ILI         0x20    /* incorrect length indicator (bit 5) */
#define   SCSI_M_SK_SKEY        0x0f    /* sense key (bit 3,2,1,0) */
#define     SCSI_E_SK_NOSENSE   0x00    /* no sense - no error */
#define     SCSI_E_SK_RECERR    0x01    /* recovered error */
#define     SCSI_E_SK_NOTRDY    0x02    /* not ready */
#define     SCSI_E_SK_MEDERR    0x03    /* medium error */
#define     SCSI_E_SK_HARDERR   0x04    /* hardware error */
#define     SCSI_E_SK_ILLREQ    0x05    /* illegal request */
#define     SCSI_E_SK_UNITATTN  0x06    /* unit attention */
#define     SCSI_E_SK_DATAPROT  0x07    /* data protection */
#define     SCSI_E_SK_CPYABORT  0x0a    /* copy/compare abort */
#define     SCSI_E_SK_CMDABORT  0x0b    /* command abort */
#define     SCSI_E_SK_MISCOMP   0x0e    /* miscompare */
        ulong info;                     /* information */
        uchar senlen;                   /* additional sense length */
        ushort copy;                    /* copy/compare command */
        ushort reserved1;               /* reserved */
        uchar sencode;                  /* additional sense code */
        uchar reserved2;                /* reserved */
        uchar fru;                      /* field replaceable unit */
        uchar bpv;                      /* bit pointer */
#define   SCSI_M_BP_FPV         0x80    /* field pointer valid */
#define   SCSI_M_BP_CD          0x40    /* command/data */
#define   SCSI_M_BP_BPV         0x10    /* bit pointer valid */
#define   SCSI_M_BP_BP          0x07    /* bit pointer */
        ushort field;                   /* field pointer */
        uchar filler[2];                /* make it multiple of 4 */
};

/*---------------------------- START/STOP UNIT -------------------------------
/ for start/stop unit command, the scsi return code should be examined.  If
/ it is CHECK CONDITION, call REQUEST SENSE command to determine the exact
/ cause of the error.
/---------------------------------------------------------------------------*/

/*----------------------------- TEST UNIT READY ------------------------------
/ For the TEST UNIT READY command, the scsi return code should be examined.
/ If it is CHECK CONDITION, REQUEST SENSE command should be issued to get
/ the full error condition.
/---------------------------------------------------------------------------*/

/*------------------------------- End of scsi.h ----------------------------*/
