#include "edtccpu.h"
#include "edtdisksect.h"
#include "edt.h"
#include "edtdevstr.h"
#include "edtdevdq.h"
#include "edtstruct.h"
#include "edtdtc.h"
#include "edtptm.h"
#include "edterror.h"
#include "edtextrn.h"
#include "vreg.h"


/*
	Interrupt is disabled when entering this routine
	If req has "wait for completion" option, then interrupt
		is enabled on return
	Insure the unit is selected and ready before the seek

	If unitstatus returns not on cyl, or there's a seek error,
		rezero drive, and set rc1

	If unit status returns device check bit set, send
		fault clear command, and set rc1
	If unit is not ready then set rc1

	Then setup cylinder to seek
	if don't wait for seek completed, return 0, interrupt still disabled

	If waiting for seek completion, then enable interrupt and wait 
		after seek is completed, check seek error
	
 */
aseek (dev, dkptr, oldpri)
register struct devdq *dev;
register struct dkinf *dkptr;
register oldpri;
{
	register retcode; register unsigned short status;
	register struct smdwr *wr; register struct smdrd *rd;

	if ( bddesc.tas1 & STOPDKIO )
		stop();

/* DDDDDDDDDDDDDDDDDDDDDDD */
	if ( dev->flag & 0x8000 ) 
		errflag(dev);
/* DDDDDDDDDDDDDDDDDDDDDDD */
	dkptr->curcyl = -1;
	*(short *)&dev->rc1 = 0;

	wr = (struct smdwr *)DK_BUSW + dev->devnum;

	/* insure drive is selected and ready to operate */

	if ( retcode = unitsel (dev->devnum) ) {

		if ( retcode == UNITNRDY )
			dev->rc1 = DER_NRDY;
		else if ( (retcode == CYLNON) || (retcode == SEEKER ) ) {
			dkrezero (dev->devnum, dkptr);
			dev->rc1 = DER_CYLNON;
		}
	
		else if ( retcode == DEVCK )  {
			wr->control = FAULTCLR;
			dev->rc1 = DER_DEVCK;
		}
	}
	if ( dev->rc1 ) {
		printrc (dev);
		dev->flag |= OPRDONE;
		if ( dev->flag & TNOWAIT )
			return(-1);		/* inturpt is disable */

		/* need to inturrpt enabled */
		splx(oldpri);
		return(-1);
	}


	/* ready to seek */
	dkptr->next = dev;
	dkptr->pd_opr = DK_SEEK;

	/* ************************************ */
	/* program clock to monitor             */
	/* if timeout, need to reset disk drive */
	dkptr->pd_tick =  TICK_1SEC;
	/* ************************************ */

	/* ******************************************* */
	/* following code should never happen, because */
	/* unitsel() routine has already checked to see if the disk is ok  */
	/* ******************************************* */
	rd =(struct smdrd *)wr;
	status = rd->unitstus & 0x0f;
	if ( (status != 0x03)) {
		printf ("s1");
	}
	if ( (*DDC_SR & 0x80) != 0x80 ) {
		printf ("s2");
	}

	/* seek a cylinder */
	wr->cylinder = dev->q_devun.pdisk.cyl;
	wr->seekend = ENSEEKEND;


	/* only return if send main memory and want to be don't wait */
	if ( (dev->flag & (DEST_L+TNOWAIT)) == TNOWAIT )
		return(0);		/* inturrpt is disable when return */

	/* allow seek interrupt to happen */
	splx(oldpri);

	/* wait seek done on specify drive */
	while ( (dkptr->pd_opr & DK_SKDONE) == 0 );

	/* must be diskio wait to be completed */
	/* set error code if seek error */

	oldpri = spl6();
	retcode = checkseek(dev, dkptr);
        dkptr->next = dev->forward;
	splx(oldpri);

	return (retcode);	/* inturpt is enabled, return 0 or -1 */
}


/* inturpt is disabled where enter this routine */

checkseek(dev, dkptr)
register struct devdq *dev;
register struct dkinf *dkptr;
{
	register unsigned short status;
	register struct smdwr *wr; register struct smdrd *rd;

	dkptr->pd_opr = 0;

	/* ********************** */
	/* turn off tick for disk */
	/* ********************** */
	dkptr->pd_tick = 0;

	rd = (struct smdrd *)DK_STSR + dev->devnum;

	/* insure seek end , cyl on, drive ready */
	while ( ((status = rd->unitstus) & SEEKEND) == 0 );


	/* after seek, unit ready (bit 0) , cyl on  (bit 1)= true, 
			seek error (bit 2), device check (bit 3) = false */
 
	if ( (status & SEEKMASK) == SEEKOK ) {
		dkptr->curcyl = dev->q_devun.pdisk.cyl;
		return (0);
	}

	if ( (status & UNITNRDY) == 0 ) 
		dev->rc1 = DER_NRDY;
	/* check error after seek */
	else if ( (status & SEEKER) || ((status & CYLNON) == 0) ) {
		dkrezero (dev->devnum, dkptr);
		dev->rc1 = DER_SEEK;
	}
	else if ( status & DEVCK ) {
		wr = (struct smdwr *)DK_BUSW + dev->devnum;
		wr->control = FAULTCLR;
		dev->rc1 = DER_DEVCK;
	}
	else
		/* don't know what error, put seek error anyway */
		dev->rc1 = DER_SEEK;

	if ( PRINT1 )
		printf ("s5(%x)", status);
	printrc (dev);
	return (-1);
}

/*
	check the status word off the physical drive
 output:
	if unit is selected and ready, return 0
	if unit not ready , return 1(UNITNRDY)
	if cylinder not on, return 2(CYLNON)
	if seeker bit is set, return 4(SEEKER)
	if device check is set, return 8(DEVCK)
	
 */
unitsel (drive)
int drive;
{
	register struct smdrd *p;
	register unsigned short status;

	p = (struct smdrd *)DK_STSR + drive;

	drivein = 0;
	/* read unit status off the physical drive */
	status = p->unitstus;

	/* check if this cause any bus error */
	if ( drivein ) {
		drivein = 0;
		if ( PRINT1 )
			printf ("\nD=%x not ready ", drive);
		return (UNITNRDY);
	}

	if ( (status & SEEKMASK) == SEEKOK )
		return (0);

	if ( (status & UNITNRDY) == 0 )
		return (UNITNRDY);
	if ( (status & CYLNON) == 0 )
		return (CYLNON);
	if ( status & SEEKER )
		return (SEEKER);
	if ( status & DEVCK )
		return (DEVCK);
	printf ("s3(%x)",status);
	return (-1);
}

/*
	setup DDC timing ID register
	input:	pointer to integer array containing timing data
 */
setimid( ptr)
register char *ptr;
{ 
	register char *timingreg;
	register int idcount;

	timingreg = DDC_TR;

	idcount = TIMIDCNT;
	while ( idcount-- ){
		*timingreg = *ptr++;
		timingreg+=2;
	}
}

/* *********************************** */
/* rezero will wait for seek completed */
/* inturpt is disabled */
/* *********************************** */

dkrezero (drive,dkptr)
register char drive;
register struct dkinf *dkptr;
{
	register struct smdwr *wr; register struct smdrd *rd;
	register i, timeout;
	register unsigned short status;

	dkptr->curcyl = -1;
	dkptr->pd_tick = 0;
	dkptr->pd_opr = 0;

	printf ("dkrezD=%x ", drive);


	wr = (struct smdwr *)DK_BUSW + drive;
	wr->control = REZERO | FAULTCLR;
	wr->seekend = ENSEEKEND;

	rd = (struct smdrd *)wr;
	
	/* put in 600ms (600000 us) time out */
	
	timeout = 1;
	for (i=0; i<300000; i++) { 
		if ( (status = rd->unitstus) & SEEKEND ) {
			timeout = 0;
			break;
		}
	}

	/* disable 'enable seek end' interrupt bit */
	wr->seekend = 0;

	if ( timeout ) {
		printf ("s4");
		return(-1);
	}

	/* insure SEEKEND, and CYLON, and UNITREADY */
	while ( ((status = rd->unitstus) & SEEKEND) == 0 );

	if ( (status & SEEKMASK) == SEEKOK )
		return (0);

	printf (" REZer(%x)", status);
	return (-1);
}
