// cpu2.c - Internal SR/DR, MMRx, APR (PAR/PDR), and CPU registers
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "pdp11/defs.h"
#include "pdp11/uqba.h"

extern uint32 p11_dsMask[];

// I/O - CPU Registers
//
// Registers  CSR Addresses  # of Registers
// APRs          772200         64 (0200)
// MMR3          772516          1 (0002)
// SR/MMR0-2     777570          4 (0010)
// APRs          777600         32 (0100)
// CPU           777740         16 (0040)

// CSR Addresses
#define APR1_CSRADR  0772200
#define MMR3_CSRADR  0772516
#define SRMM_CSRADR  0777570
#define APR2_CSRADR  0777600
#define CPU_CSRADR   0777740

// Number of Registers
#define APR1_NREGS   (0200 >> 1)  // 64 Registers
#define MMR3_NREGS   (0002 >> 1)  //  1 Register
#define SRMM_NREGS   (0010 >> 1)  //  4 Registers
#define APR2_NREGS   (0100 >> 1)  // 32 Registers
#define CPU_NREGS    (0040 >> 1)  // 16 Registers

// Memory Management Registers

#ifdef DEBUG
static cchar *smNameR[] = {
	"SR",     // Switch Register
	"MMR0",   // Memory Management Register #0 - Status Register
	"MMR1",   // Memory Management Register #1 - R+/-R Recovery
	"MMR2",   // Memory Management Register #2 - Saved PC
};
static cchar *smNameW[] = {
	"DR",     // Display Register
	"MMR0",   // Memory Management Register #0 - Status Register
	"MMR1",   // Memory Management Register #1 - R+/-R Recovery
	"MMR2",   // Memory Management Register #2 - Saved PC
};
#endif /* DEBUG */

int p11_ReadSRMM(void *dptr, uint32 pAddr, uint16 *data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;
	int     reg  = (pAddr >> 1) & 3;

	switch (reg) {
		case 0: // SR - Switch Register
			*data = SR;
			break;

		case 1: // SR0/MMR0 - Status Register
			*data = MMR0;
			break;

		case 2: // SR1/MMR1 - R+/-R Recovery
			*data = MMR1;
			break;

		case 3: // SR2/MMR2 - Saved PC
			*data = MMR2;
			break;
	}

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: %s (%o) => %06o\n",
			p11->Unit.devName, smNameR[reg], pAddr, *data);
#endif /* DEBUG */

	return UQ_OK;
}

int p11_WriteSRMM(void *dptr, uint32 pAddr, uint16 data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;
	int     reg  = (pAddr >> 1) & 3;

	switch (reg) {
		case 0: // DR - Display Register
			DR = data;
			break;

		case 1: // SR0/MMR0: Status Register
			if (acc == ACC_BYTE) // If byte access, merge data with MMR0.
				data = (pAddr & 1) ? ((data << 8)   | (MMR0 & 0377)) :
				                     ((data & 0377) | (MMR0 & ~0377));
			MMR0 = (data & MMR0_RW) | (MMR0 & ~MMR0_RW);
			break;

		default: // Otherwise - Read-only Registers.
			break;
	}

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: %s (%o) <= %06o (Now: %06o)\n",
			p11->Unit.devName, smNameR[reg], pAddr, data, (reg == 0) ? DR :
			(reg == 1) ? MMR0 : (reg == 2) ? MMR1 : MMR2);
#endif /* DEBUG */

	return UQ_OK;
}

int p11_ReadMMR3(void *dptr, uint32 pAddr, uint16 *data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;

	*data = MMR3;

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: (R) MMR3 (%o) => %06o\n",
			p11->Unit.devName, pAddr, *data);
#endif /* DEBUG */

	return UQ_OK;
}

int p11_WriteMMR3(void *dptr, uint32 pAddr, uint16 data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;

	if ((pAddr & 1) == 0) {
		MMR3 = data & MMR3_RW;
	}

	// Update Data Space.
	DSPACE = GetDSpace(PSW_GETCUR(PSW));

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: (W) MMR3 (%o) <= %06o (Now: %06o)\n",
			p11->Unit.devName, pAddr, data, MMR3);
#endif /* DEBUG */

	return UQ_OK;
}

// PAR/PDR Registers
// 
// 772200-772277  - Supervisor Block
// 772300-772377  - Kernel Block
// 777600-777677  - User Block

int p11_ReadAPR(void *dptr, uint32 pAddr, uint16 *data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;
	uint16  idx, left;

	// Convert I/O address to APR index.
	idx  = (pAddr >> 1) & 017;
	left = (pAddr >> 5) & 1;
	if ((pAddr & 0100) == 0)
		idx |= 020;
	if (pAddr & 0400)
		idx |= 040;

	// Get PAR or PDR from APR table.
	*data = left ? (APR(idx) >> 16) : APR(idx);

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: APR %03o (%o) => %06o (%06o %06o)\n", 
			p11->Unit.devName, idx, pAddr, *data,
			APR(idx) >> 16, APR(idx) & 0177777);
#endif /* DEBUG */

	return UQ_OK;
}

int p11_WriteAPR(void *dptr, uint32 pAddr, uint16 data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;
	uint16  idx, left, apr;

	// Convert I/O address to APR index.
	idx  = (pAddr >> 1) & 017;
	left = (pAddr >> 5) & 1;
	if ((pAddr & 0100) == 0)
		idx |= 020;
	if (pAddr & 0400)
		idx |= 040;

	if (left) {
		// PAR (Processor Address Register)
		if (acc == ACC_BYTE) {
			// If byte access, merge data with APR entry.
			apr  = APR(idx) >> 16;
			data = (pAddr & 1) ? ((data << 8)   | (apr & 0377)) :
		   	                  ((data & 0377) | (apr & ~0377));
		}
		APR(idx) = (data << 16) | (APR(idx) & (0177777 & ~PDR_W));
	} else {
		// PDR (Processor Descriptor Register)
		if (acc == ACC_BYTE) {
			// If byte access, merge data with APR entry.
			apr = APR(idx);
			data = (pAddr & 1) ? ((data << 8)   | (apr & 0377)) :
		   	                  ((data & 0377) | (apr & ~0377));
		}
		APR(idx) = (data & PDR_RW) | (APR(idx) & ~(PDR_RW|PDR_W));
	}

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: APR %03o (%o) <= %06o (%06o %06o)\n", 
			p11->Unit.devName, idx, pAddr, data,
			APR(idx) >> 16, APR(idx) & 0177777);
#endif /* DEBUG */

	return UQ_OK;
}

// CPU Registers - 777740 to 777777
//
// MEMERR  777744  Memory Error Register  Read Only/Write to Reset
// CCR     777746  Cache Control Register Read/Write
// MAINT   777750  Maintenance Register   Read Only
// HITMISS 777752  His/Miss Register      Read Only
// CPUERR  777766  CPU Error Register     Read Only/Write to Reset
// PIRQ    777772  Priority Interrupts    Read/Write with Side Effects
// PSW     777776  Process Status Word    Read/Write with Side Effects

p11_ReadCPU(void *dptr, uint32 pAddr, uint16 *data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;

	switch ((pAddr >> 1) & 017) {
		case 002: // Memory Error (MEMERR)
			*data = MEMERR;
			break;
		case 003: // Cache Control Register (CCR)
			*data = CCR;
			break;
		case 004: // Maintenance Register (MAINT)
			*data = MAINT;
			break;
		case 005: // Hit/Miss Register (HMR)
			*data = HMR;
			break;
		case 013: // CPU Error Register (CPUERR)
			*data = CPUERR;
			break;
		case 015: // Priority Interrupt Requests (PIRQ)
			*data = PIRQ;
			break;
		case 017: // Processor Status Word (PSW)
			*data = PSW|CC;
			break;
		default:  // Otherwise - Non-existant Memory
			*data = 0;
			return UQ_NXM;
	}

	if (acc & ACC_INST)
		return UQ_ADRERR;
	return UQ_OK;
}

p11_WriteCPU(void *dptr, uint32 pAddr, uint16 data, uint32 acc)
{
	P11_CPU *p11 = (P11_CPU *)dptr;

	switch ((pAddr >> 1) & 017) {
		case 002: // Memory Error Register
			MEMERR = 0;
			return UQ_OK;

		case 003: // Cache Control Register
			if (acc == ACC_BYTE) // If byte access, merge data with CCR register.
				data = (pAddr & 1) ? ((data << 8)   | (CCR & 0377)) :
		                           ((data & 0377) | (CCR & ~0377));
			CCR = data;
			return UQ_OK;

		case 004: // Maintenance Register
			// Read-only Register - Do nothing
			return UQ_OK;

		case 005: // Hit/Miss Register
			// Read-only Register - Do nothing
			return UQ_OK;

		case 013: // CPU Error Register
			CPUERR = 0;
			return UQ_OK;

		case 015: // Priority Interrupt Requests
			{
				uint16 pl = 0;

				if (acc == ACC_BYTE) {
					if (pAddr & 1) data <<= 8;
					else           return UQ_OK;
				}
				PIRQ = data & PIRQ_RW;

				if (PIRQ & PIRQ_PIR1) pl = 0042;
				if (PIRQ & PIRQ_PIR2) pl = 0104;
				if (PIRQ & PIRQ_PIR3) pl = 0146;
				if (PIRQ & PIRQ_PIR4) pl = 0210;
				if (PIRQ & PIRQ_PIR5) pl = 0252;
				if (PIRQ & PIRQ_PIR6) pl = 0314;
				if (PIRQ & PIRQ_PIR7) pl = 0356;
				PIRQ |= pl;

				// Update interrupt requests.
				uq11_EvalIRQ(p11, GET_IPL(PSW));
			}
			return UQ_OK;

		case 017: // Processor Status Word
			{
				uint16 newPSW,  oldPSW = PSW;
				uint16 newMode, oldMode;
				int    idx;

				if (acc == ACC_BYTE) // If byte access, merge data with APR entry.
					data = (pAddr & 1) ? ((data << 8)   | (PSW & 0377)) :
		         	                  ((data & 0377) | (PSW & ~0377));
				newPSW = (data & PSW_RW) | (PSW & ~PSW_RW);

				// Get modes from old and new PSW.
				oldMode = PSW_GETCUR(oldPSW);
				newMode = PSW_GETCUR(newPSW);

				// Switch SP between kernel, supervisor, or user.
				// Swap two register sets if new RS mode request.
				STKREG(oldMode) = SP;
				if ((oldPSW ^ newPSW) & PSW_RS) {
					uint32 oldSet = (oldPSW & PSW_RS) ? 1 : 0;
					uint32 newSet = (newPSW & PSW_RS) ? 1 : 0;

					// Exchange all registers R0-R5 with R0'-R5'
					for (idx = 0; idx < 6; idx++) {
						GPREG(idx, oldSet) = REGW(idx);
						REGW(idx) = GPREG(idx, newSet);
					}
				}
				SP  = STKREG(newMode);
				PSW = newPSW & ~CC_ALL;
				CC  = newPSW & CC_ALL;

				// Set instruction/data space for memory management.
				ISPACE = GetISpace(newMode);
				DSPACE = GetDSpace(newMode);

				// Update interrupt requests.
				uq11_EvalIRQ(p11, GET_IPL(PSW));
			}
			return UQ_OK;
	}

	return UQ_NXM;
}

// Set up internal I/O registers for MMRx, APR, and CPU registers
void p11_InitCpuRegs(register P11_CPU *p11)
{
	UQ_IO  *uq = (UQ_IO *)p11->uqba;
	MAP_IO *io;

	// Set up CPU registers in I/O map area.
	io             = &uq->ioCPU;
	io->devName    = p11->Unit.devName;
	io->keyName    = p11->Unit.keyName;
	io->emuName    = "CPU Registers";
	io->emuVersion = NULL;
	io->Device     = p11;
	io->csrAddr    = CPU_CSRADR;
	io->nRegs      = CPU_NREGS;
	io->ReadIO     = p11_ReadCPU;
	io->WriteIO    = p11_WriteCPU;
	uq->Callback->SetMap(uq, io);

	// Set up SR/MMR0-2 registers in I/O map area.
	io             = &uq->ioSRMM;
	io->devName    = p11->Unit.devName;
	io->keyName    = p11->Unit.keyName;
	io->emuName    = "SR/MMR0-2 Registers";
	io->emuVersion = NULL;
	io->Device     = p11;
	io->csrAddr    = SRMM_CSRADR;
	io->nRegs      = SRMM_NREGS;
	io->ReadIO     = p11_ReadSRMM;
	io->WriteIO    = p11_WriteSRMM;
	uq->Callback->SetMap(uq, io);

	// Set up MMR3 register in I/O map area.
	io             = &uq->ioMMR3;
	io->devName    = p11->Unit.devName;
	io->keyName    = p11->Unit.keyName;
	io->emuName    = "MMR3 Register";
	io->emuVersion = NULL;
	io->Device     = p11;
	io->csrAddr    = MMR3_CSRADR;
	io->nRegs      = MMR3_NREGS;
	io->ReadIO     = p11_ReadMMR3;
	io->WriteIO    = p11_WriteMMR3;
	uq->Callback->SetMap(uq, io);

	// Set up APR (Kernel/Supervisor) registers in I/O map area.
	io             = &uq->ioAPR1;
	io->devName    = p11->Unit.devName;
	io->keyName    = p11->Unit.keyName;
	io->emuName    = "APR Registers (Kern, Super)";
	io->emuVersion = NULL;
	io->Device     = p11;
	io->csrAddr    = APR1_CSRADR;
	io->nRegs      = APR1_NREGS;
	io->ReadIO     = p11_ReadAPR;
	io->WriteIO    = p11_WriteAPR;
	uq->Callback->SetMap(uq, io);

	// Set up APR (User) registers in I/O map area.
	io             = &uq->ioAPR2;
	io->devName    = p11->Unit.devName;
	io->keyName    = p11->Unit.keyName;
	io->emuName    = "APR Registers (User)";
	io->emuVersion = NULL;
	io->Device     = p11;
	io->csrAddr    = APR2_CSRADR;
	io->nRegs      = APR2_NREGS;
	io->ReadIO     = p11_ReadAPR;
	io->WriteIO    = p11_WriteAPR;
	uq->Callback->SetMap(uq, io);

	p11->InitTimer(p11);
}
