// unit.c - TS10 Device Mapping routines
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "emu/defs.h"

void *kxio_Create(char *, int, char **);

#define TS10_MAXDEVICES 256

typedef struct MapDevice2 MAP_DEVICE;
typedef struct MapDeviceInfo MAP_DINFO;

struct MapDevice2 {
	char  *Name;
	DINFO *DevInfo;
	void  *Device;
};

struct MapDeviceInfo {
	DINFO *DevInfo;
};

MAP_DEVICE ts10_MapDevices[TS10_MAXDEVICES];

boolean ts10_CreateDevice(char *newName, void *newDevice)
{
	int idx;

	for (idx = 0; idx < TS10_MAXDEVICES; idx++) {
		if (ts10_MapDevices[idx].Name == NULL) {
			ts10_MapDevices[idx].Device = newDevice;
			ts10_MapDevices[idx].DevInfo = ((MAP_DINFO *)newDevice)->DevInfo;
			ts10_MapDevices[idx].Name = (char *)malloc(strlen(newName)+1);
			strcpy(ts10_MapDevices[idx].Name, newName);
			return FALSE;
		}
	}
	return TRUE;
}

MAP_DEVICE *ts10_FindDevice(char *fndName)
{
	int idx;

	for (idx = 0; idx < TS10_MAXDEVICES; idx++) {
		MAP_DEVICE *map = &ts10_MapDevices[idx];
		if (map->Name && !strcasecmp(fndName, map->Name))
			return &ts10_MapDevices[idx];
	}

	return NULL;
}

MAP_DEVICE *ts10_GetDevice(int argc, char **argv)
{
	MAP_DEVICE *map;

	if (argc < 3) {
		printf("%s: Too few arguments.\n", argv[0]);
		printf("Usage: %s <device> ...\n", argv[0]);
		return NULL;
	}

	if ((map = ts10_FindDevice(argv[1])) == NULL) {
		printf("%s: Device not found\n", argv[1]);
		return NULL;
	}

	return map;
}

// Usage: create <device> ...
int ts10_CmdCreate(int argc, char **argv)
{
	void *dptr;

	// Make ensure least three arguments 
	if (argc < 3) {
		printf("%s: Too few arguments\n", argv[0]);
		printf("Usage: %s <device> ...\n", argv[0]);
		return EMU_OK;
	}

	// Check if device name taken or already created first.
	RemoveSpaces((char *)argv[1]);
	if (ts10_FindDevice(argv[1])) {
		printf("%s: Device name taken or already created.\n", argv[1]);
		return EMU_OK;
	}

	RemoveSpaces((char *)argv[2]);
	dptr = kxio_Create(argv[2], argc-3, &argv[3]);

	ts10_CreateDevice(argv[1], dptr);

	return EMU_OK;
}

// Usage: attach <device> ...
int ts10_CmdAttach(int argc, char **argv)
{
	MAP_DEVICE *map;

	if (map = ts10_GetDevice(argc, argv)) {
		if (map->DevInfo->Attach) {
			// Call Device's Attach routine to complete operation.
			return map->DevInfo->Attach(map->Device, argc-2, &argv[2]);
		} else {
			printf("%s: Not supported.\n", argv[0]);
		}
	}

	return EMU_OK;
}

// Usage: detach <device> ...
int ts10_CmdDetach(int argc, char **argv)
{
	MAP_DEVICE *map;

	if (map = ts10_GetDevice(argc, argv)) {
		if (map->DevInfo->Detach) {
			// Call Device's Detach routine to complete operation.
			return map->DevInfo->Detach(map->Device, argc-2, &argv[2]);
		} else {
			printf("%s: Not Supported.\n", argv[0]);
		}
	}

	return EMU_OK;
}

// Usage: info <device> ...
int ts10_CmdInfo(int argc, char **argv)
{
	MAP_DEVICE *map;

	if (map = ts10_GetDevice(argc, argv)) {
		if (map->DevInfo->Info) {
			// Call Device's Info routine to complete operation.
			return map->DevInfo->Info(map->Device, argc-2, &argv[2]);
		} else {
			printf("%s: Not Supported.\n", argv[0]);
		}
	}

	return EMU_OK;
}
