// m6502.c - Motorola 6502 Microprocessor Main Routines
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "emu/defs.h"
#include "cpu/m6502/m6502.h"

#define RD_MEM(addr)       m65->ReadB(addr)
#define WR_MEM(addr, data) m65->WriteB(addr, data)

#define PUSH(data)  WR_MEM(SP, data); S--
#define PULL(data)  S++; data = RD_MEM(SP)

#define RD_OP   RD_MEM(PC++)

#define EA_ABS   EAL = RD_OP; EAH = RD_OP
#define EA_ABX   EA_ABS; EA += X
#define EA_ABY   EA_ABS; EA += Y
#define EA_ZPG   ZPL = RD_OP;     EA = ZP
#define EA_ZPX   ZPL = RD_OP + X; EA = ZP
#define EA_ZPY   ZPL = RD_OP + Y; EA = ZP
#define EA_IDX   ZPL = RD_OP + X; EAL = RD_MEM(ZP); ZPL++; EAH = RD_MEM(ZP);
#define EA_IDY   ZPL = RD_OP; EAL = RD_MEM(ZP); ZPL++; EAH = RD_MEM(ZP); EA += Y;
#define EA_IND   EA_ABS; tmp = RD_MEM(EA); EAL++; EAH = RD_MEM(EA); EAL = tmp;

// Read a value from effective address area
#define RD_IMM  tmp = RD_OP
#define RD_ACC  tmp = A
#define RD_ABS  EA_ABS; tmp = RD_MEM(EA)
#define RD_ABX  EA_ABX; tmp = RD_MEM(EA)
#define RD_ABY  EA_ABY; tmp = RD_MEM(EA)
#define RD_ZPG  EA_ZPG; tmp = RD_MEM(EA)
#define RD_ZPX  EA_ZPX; tmp = RD_MEM(EA)
#define RD_ZPY  EA_ZPY; tmp = RD_MEM(EA)
#define RD_IDX  EA_IDX; tmp = RD_MEM(EA)
#define RD_IDY  EA_IDY; tmp = RD_MEM(EA)

// Write a value from temp
#define WR_ABS  EA_ABS; WR_MEM(EA, tmp)
#define WR_ABX  EA_ABX; WR_MEM(EA, tmp)
#define WR_ABY  EA_ABY; WR_MEM(EA, tmp)
#define WR_ZPG  EA_ZPG; WR_MEM(EA, tmp)
#define WR_ZPX  EA_ZPX; WR_MEM(EA, tmp)
#define WR_ZPY  EA_ZPY; WR_MEM(EA, tmp)
#define WR_IDX  EA_IDX; WR_MEM(EA, tmp)
#define WR_IDY  EA_IDY; WR_MEM(EA, tmp)

// Write back a value from temp to effective address area.
#define WB_ACC  A = tmp
#define WB_EA   WR_MEM(EA, tmp)

#define BRA(cond) \
	if (cond) { tmp = RD_OP; PC += (int8)tmp; } \
	else      PC++;

// ADC - Add with Carry
#define ADC  if (P & SR_D) { \
		  		 } else {                             \
					int sum = A + tmp + (P & SR_C);    \
					P &= ~(SR_V|SR_C);                 \
					if (~(A ^ tmp) & (A ^ sum) & SR_N) \
						P |= SR_V;                      \
					if (sum & 0xFF00)                  \
						P |= SR_C;                      \
					A = (uint8)sum; SET_NZ(A);         \
				 }

// AND - Logical AND
#define AND  A &= tmp; SET_NZ(A)

// ASL - Arithmetic Shift Left
#define ASL  P = (P & ~SR_C) | ((tmp >> 7) & SR_C); \
             tmp <<= 1; SET_NZ(tmp)

// Bcc - Branch with Condition
#define BCC  BRA((P & SR_C) == 0)   // BCC - Branch if Carry Clear
#define BCS  BRA(P & SR_C)          // BCS - Branch if Carry Set
#define BEQ  BRA(P & SR_Z)          // BEQ - Branch if Equal
#define BMI  BRA(P & SR_N)          // BMI - Branch if Minus
#define BNE  BRA((P & SR_Z) == 0)   // BNE - Branch if Not Equal
#define BPL  BRA((P & SR_N) == 0)   // BPL - Branch if Plus
#define BVC  BRA((P & SR_V) == 0)   // BVC - Branch if Overflow Clear
#define BVS  BRA(P & SR_V)          // BVS - Branch if Overflow Set

// BIT - Bit Test
#define BIT  P &= ~(SR_N|SR_V|SR_Z); \
             P |= tmp & (SR_N|SR_V); \
             if ((tmp & A) == 0) P |= SR_Z
				 
// BRK - Break
#define BRK  PC++;                   \
             PUSH(PCH); PUSH(PCL);   \
             PUSH(P | SR_B);         \
             P |= SR_I;              \
             PCL = RD_MEM(IRQ_ADR);  \
             PCH = RD_MEM(IRQ_ADR+1) 

// CLC - Clear Carry
#define CLC  P &= ~SR_C

// CLD - Clear Decimal Mode
#define CLD  P &= ~SR_D

// CLI - Clear Interrupt Disable
#define CLI  P &= ~SR_I

// CLV - Clear Overflow
#define CLV  P &= ~SR_V

// CMP - Compare Accumulator
#define CMP  P &= ~SR_C; if (A >= tmp) P |= SR_C; SET_NZ(A - tmp)

// CPX - Compare X Index Register
#define CPX  P &= ~SR_C; if (X >= tmp) P |= SR_C; SET_NZ(X - tmp)

// CPY - Compare Y Index Register
#define CPY  P &= ~SR_C; if (Y >= tmp) P |= SR_C; SET_NZ(Y - tmp)

// DEC - Decrement
#define DEC  tmp--; SET_NZ(tmp)

// DEX - Decrement X Index Register
#define DEX  X--; SET_NZ(X)

// DEY - Decrement Y Index Register
#define DEY  Y--; SET_NZ(Y)

// EOR - Exclusive OR
#define EOR  A ^= tmp; SET_NZ(A)

// INC - Increment
#define INC  tmp++; SET_NZ(tmp)

// INX - Increment X Index Register
#define INX  X++; SET_NZ(X)

// INY - Increment Y Index Register
#define INY  Y++; SET_NZ(Y)

// JMP - Jump
#define JMP  PC = EA

// JSR - Jump to Subroutine
#define JSR  EAL = RD_OP;           \
             PUS(PCH); PUSH(PCL);  \
             EAH = RD_OP;           \
             PC = EA

// LDA - Load Accumulator
//#define LDA  A = RD_MEM(EA); SET_NZ(A)
#define LDA  A = (uint8)tmp; SET_NZ(A)

// LDX - Load X Index Register
//#define LDX  X = RD_MEM(EA); SET_NZ(X)
#define LDX  X = (uint8)tmp; SET_NZ(X)

// LDY - Load Y Index Register
//#define LDY  Y = RD_MEM(EA); SET_NZ(Y)
#define LDY  Y = (uint8)tmp; SET_NZ(Y)

// LSR - Logical Shift Right
#define LSR  P = (P & ~SR_C) | (tmp & SR_C); \
             tmp >>= 1; SET_NZ(tmp)

// NOP - No Operation
#define NOP  // Do Nothing

// Logical OR Accumulator
#define ORA  A |= tmp; SET_NZ(A)

// PHA - Push Accumulator
#define PHA  PUSH(A)

// PHP - Push Processor Status
#define PHP  PUSH(P)

// PLA - Pull Accumulator
#define PLA  PULL(A); SET_NZ(A)

// PLP - Pull Processor Status
#define PLP  PULL(P); P |= (SR_T|SR_B)

// ROL - Rotate Left
#define ROL  tmp = (tmp << 1) | (P & SR_C);         \
             P = (P & ~SR_C) | ((tmp >> 8) & SR_C); \
             SET_NZ((uint8)tmp)

// ROR - Rotate Right
#define ROR  tmp |= (P & SR_C) << 8;          \
             P = (P & ~SR_C) | (tmp & SR_C);  \
             tmp >>= 1; SET_NZ((uint8)tmp)

// RTI - Return from Interrupt
#define RTI  PULL(P); PULL(PCL); PULL(PCH); P |= (SR_T|SR_B)

// RTS - Return from Subroutine
#define RTS  PULL(PCL); PULL(PCH); PC++

// SBC - Subtract with Carry
#define SBC  if (P & SR_D) { \
             } else {                             \
                int diff = A - tmp - (P & SR_C);   \
                P &= ~(SR_V|SR_C);                 \
                if ((A ^ tmp) & (A ^ diff) & SR_N) \
                   P |= SR_V;                      \
                if ((diff & 0xFF00) == 0)          \
                   P |= SR_C;                      \
                A = (uint8)diff; SET_NZ(A);        \
             }

// SEC - Set Carry
#define SEC  P |= SR_C

// SED - Set Decimal Mode
#define SED  P |= SR_D

// SEI - Set Interrupt Disable
#define SEI  P |= SR_I

// STA - Store Accumulator
//#define STA  WR_MEM(EA, A)
#define STA  tmp = (uint8)A

// STX - Store X Index Register
//#define STX  WR_MEM(EA, X)
#define STX  tmp = (uint8)X

// STY - Store Y Index Register
//#define STY  WR_MEM(EA, Y)
#define STY  tmp = (uint8)Y

// TAX - Transfer Accumulator to X Index Register
#define TAX  X = A; SET_NZ(X)

// TAY - Transfer Accumulator to Y Index Register
#define TAY  Y = A; SET_NZ(Y)

// TSX - Transfer Stack Pointer to X Index Register
#define TSX  X = S; SET_NZ(S)

// TXA - Transfer X Index Register to Accumulator
#define TXA  A = X; SET_NZ(A)

// TXS - Transfer X Index Register to Stack Pointer
#define TXS  S = X;

// TYA - Transfer Y Index Register to Accumulator
#define TYA  A = Y; SET_NZ(A)

// Illegal Instruction
#define ILL

#define CYCLES(x)

inline void m6502_ExecuteOpcode(register M65_CPU *m65, uint8 opCode)
{
	int32 tmp;

	switch(opCode) {
		case 0x00:         BRK;    CYCLES(7);  break;
		case 0x20:         JSR;    CYCLES(6);  break;
		case 0x40:         RTI;    CYCLES(6);  break;
		case 0x60:         RTS;    CYCLES(6);  break;
		case 0x80:         ILL;    CYCLES(2);  break;
		case 0xA0: RD_IMM; LDY;    CYCLES(2);  break;
		case 0xC0: RD_IMM; CPY;    CYCLES(2);  break;
		case 0xE0: RD_IMM; CPX;    CYCLES(2);  break;
	
		case 0x10:         BPL;                break;
		case 0x30:         BMI;                break;
		case 0x50:         BVC;                break;
		case 0x70:         BVS;                break;
		case 0x90:         BCC;                break;
		case 0xB0:         BCS;                break;
		case 0xD0:         BNE;                break;
		case 0xF0:         BEQ;                break;
	
		case 0x01: RD_IDX; ORA;          CYCLES(6); break;
		case 0x21: RD_IDX; AND;          CYCLES(6); break;
		case 0x41: RD_IDX; EOR;          CYCLES(6); break;
		case 0x61: RD_IDX; ADC;          CYCLES(6); break;
		case 0x81:         STA; WR_IDX;  CYCLES(6); break;
		case 0xA1: RD_IDX; LDA;          CYCLES(6); break;
		case 0xC1: RD_IDX; CMP;          CYCLES(6); break;
		case 0xE1: RD_IDX; SBC;          CYCLES(6); break;

		case 0x11: RD_IDY; ORA;          CYCLES(5); break;
		case 0x31: RD_IDY; AND;          CYCLES(5); break;
		case 0x51: RD_IDY; EOR;          CYCLES(5); break;
		case 0x71: RD_IDY; ADC;          CYCLES(5); break;
		case 0x91:         STA; WR_IDY;  CYCLES(6); break;
		case 0xB1: RD_IDY; LDA;          CYCLES(5); break;
		case 0xD1: RD_IDY; CMP;          CYCLES(5); break;
		case 0xF1: RD_IDY; SBC;          CYCLES(5); break;
	
		case 0x02:         ILL;          CYCLES(2); break;
		case 0x22:         ILL;          CYCLES(2); break;
		case 0x42:         ILL;          CYCLES(2); break;
		case 0x62:         ILL;          CYCLES(2); break;
		case 0x82:         ILL;          CYCLES(2); break;
		case 0xA2: RD_IMM; LDX;          CYCLES(2); break;
		case 0xC2:         ILL;          CYCLES(2); break;
		case 0xE2:         ILL;          CYCLES(2); break;
	
		case 0x12:         ILL;          CYCLES(2); break;
		case 0x32:         ILL;          CYCLES(2); break;
		case 0x52:         ILL;          CYCLES(2); break;
		case 0x72:         ILL;          CYCLES(2); break;
		case 0x92:         ILL;          CYCLES(2); break;
		case 0xB2:         ILL;          CYCLES(2); break;
		case 0xD2:         ILL;          CYCLES(2); break;
		case 0xF2:         ILL;          CYCLES(2); break;
		
		case 0x03:         ILL;          CYCLES(2); break;
		case 0x23:         ILL;          CYCLES(2); break;
		case 0x43:         ILL;          CYCLES(2); break;
		case 0x63:         ILL;          CYCLES(2); break;
		case 0x83:         ILL;          CYCLES(2); break;
		case 0xA3:         ILL;          CYCLES(2); break;
		case 0xC3:         ILL;          CYCLES(2); break;
		case 0xE3:         ILL;          CYCLES(2); break;
	
		case 0x13:         ILL;          CYCLES(2); break;
		case 0x33:         ILL;          CYCLES(2); break;
		case 0x53:         ILL;          CYCLES(2); break;
		case 0x73:         ILL;          CYCLES(2); break;
		case 0x93:         ILL;          CYCLES(2); break;
		case 0xB3:         ILL;          CYCLES(2); break;
		case 0xD3:         ILL;          CYCLES(2); break;
		case 0xF3:         ILL;          CYCLES(2); break;
		
		case 0x04:         ILL;          CYCLES(2); break;
		case 0x24: RD_ZPG; BIT;          CYCLES(3); break;
		case 0x44:         ILL;          CYCLES(2); break;
		case 0x64:         ILL;          CYCLES(2); break;
		case 0x84:         STY; WR_ZPG;  CYCLES(3); break;
		case 0xA4: RD_ZPG; LDY;          CYCLES(3); break;
		case 0xC4: RD_ZPG; CPY;          CYCLES(3); break;
		case 0xE4: RD_ZPG; CPX;          CYCLES(3); break;
	
		case 0x14:         ILL;          CYCLES(2); break;
		case 0x34:         ILL;          CYCLES(2); break;
		case 0x54:         ILL;          CYCLES(2); break;
		case 0x74:         ILL;          CYCLES(2); break;
		case 0x94:         STY; WR_ZPX;  CYCLES(4); break;
		case 0xB4: RD_ZPX; LDY;          CYCLES(4); break;
		case 0xD4:         ILL;          CYCLES(2); break;
		case 0xF4:         ILL;          CYCLES(2); break;

		case 0x05: RD_ZPG; ORA;          CYCLES(3); break;
		case 0x25: RD_ZPG; AND;          CYCLES(3); break;
		case 0x45: RD_ZPG; EOR;          CYCLES(3); break;
		case 0x65: RD_ZPG; ADC;          CYCLES(3); break;
		case 0x85:         STA; WR_ZPG;  CYCLES(3); break;
		case 0xA5: RD_ZPG; LDA;          CYCLES(3); break;
		case 0xC5: RD_ZPG; CMP;          CYCLES(3); break;
		case 0xE5: RD_ZPG; SBC;          CYCLES(3); break;
	
		case 0x15: RD_ZPX; ORA;          CYCLES(4); break;
		case 0x35: RD_ZPX; AND;          CYCLES(4); break;
		case 0x55: RD_ZPX; EOR;          CYCLES(4); break;
		case 0x75: RD_ZPX; ADC;          CYCLES(4); break;
		case 0x95:         STA; WR_ZPX;  CYCLES(4); break;
		case 0xB5: RD_ZPX; LDA;          CYCLES(4); break;
		case 0xD5: RD_ZPX; CMP;          CYCLES(4); break;
		case 0xF5: RD_ZPX; SBC;          CYCLES(4); break;
	
		case 0x06: RD_ZPG; ASL; WB_EA;   CYCLES(5); break;
		case 0x26: RD_ZPG; ROL; WB_EA;   CYCLES(5); break;
		case 0x46: RD_ZPG; LSR; WB_EA;   CYCLES(5); break;
		case 0x66: RD_ZPG; ROR; WB_EA;   CYCLES(5); break;
		case 0x86:         STX; WR_ZPG;  CYCLES(3); break;
		case 0xA6: RD_ZPG; LDX;          CYCLES(3); break;
		case 0xC6: RD_ZPG; DEC; WB_EA;   CYCLES(5); break;
		case 0xE6: RD_ZPG; INC; WB_EA;   CYCLES(5); break;

		case 0x16: RD_ZPX; ASL; WB_EA;   CYCLES(6); break;
		case 0x36: RD_ZPX; ROL; WB_EA;   CYCLES(6); break;
		case 0x56: RD_ZPX; LSR; WB_EA;   CYCLES(6); break;
		case 0x76: RD_ZPX; ROR; WB_EA;   CYCLES(6); break;
		case 0x96:         STX; WR_ZPY;  CYCLES(4); break;
		case 0xB6: RD_ZPY; LDX;          CYCLES(4); break;
		case 0xD6: RD_ZPX; INC; WB_EA;   CYCLES(6); break;
		case 0xF6: RD_ZPX; DEC; WB_EA;   CYCLES(6); break;
		
		case 0x07:         ILL;          CYCLES(2); break;
		case 0x27:         ILL;          CYCLES(2); break;
		case 0x47:         ILL;          CYCLES(2); break;
		case 0x67:         ILL;          CYCLES(2); break;
		case 0x87:         ILL;          CYCLES(2); break;
		case 0xA7:         ILL;          CYCLES(2); break;
		case 0xC7:         ILL;          CYCLES(2); break;
		case 0xE7:         ILL;          CYCLES(2); break;
		
		case 0x17:         ILL;          CYCLES(2); break;
		case 0x37:         ILL;          CYCLES(2); break;
		case 0x57:         ILL;          CYCLES(2); break;
		case 0x77:         ILL;          CYCLES(2); break;
		case 0x97:         ILL;          CYCLES(2); break;
		case 0xB7:         ILL;          CYCLES(2); break;
		case 0xD7:         ILL;          CYCLES(2); break;
		case 0xF7:         ILL;          CYCLES(2); break;
		
		case 0x08:         PHP;          CYCLES(2); break;
		case 0x28:         PLP;          CYCLES(2); break;
		case 0x48:         PHA;          CYCLES(2); break;
		case 0x68:         PLA;          CYCLES(2); break;
		case 0x88:         DEY;          CYCLES(2); break;
		case 0xA8:         TAY;          CYCLES(2); break;
		case 0xC8:         INY;          CYCLES(2); break;
		case 0xE8:         INX;          CYCLES(2); break;
	
		case 0x18:         CLC;          CYCLES(2); break;
		case 0x38:         SEC;          CYCLES(2); break;
		case 0x58:         CLI;          CYCLES(2); break;
		case 0x78:         SEI;          CYCLES(2); break;
		case 0x98:         TYA;          CYCLES(2); break;
		case 0xB8:         CLV;          CYCLES(2); break;
		case 0xD8:         CLD;          CYCLES(2); break;
		case 0xF8:         SED;          CYCLES(2); break;

		case 0x09: RD_IMM; ORA;          CYCLES(2); break;
		case 0x29: RD_IMM; AND;          CYCLES(2); break;
		case 0x49: RD_IMM; EOR;          CYCLES(2); break;
		case 0x69: RD_IMM; ADC;          CYCLES(2); break;
		case 0x89:         ILL;          CYCLES(2); break;
		case 0xA9: RD_IMM; LDA;          CYCLES(2); break;
		case 0xC9: RD_IMM; CMP;          CYCLES(2); break;
		case 0xE9: RD_IMM; SBC;          CYCLES(2); break;
	
		case 0x19: RD_ABY; ORA;          CYCLES(4); break;
		case 0x39: RD_ABY; AND;          CYCLES(4); break;
		case 0x59: RD_ABY; EOR;          CYCLES(4); break;
		case 0x79: RD_ABY; ADC;          CYCLES(4); break;
		case 0x99:         STA; WR_ABY;  CYCLES(5); break;
		case 0xB9: RD_ABY; LDA;          CYCLES(4); break;
		case 0xD9: RD_ABY; CMP;          CYCLES(4); break;
		case 0xF9: RD_ABY; SBC;          CYCLES(4); break;
	
		case 0x0A: RD_ACC; ASL; WB_ACC;  CYCLES(2); break;
		case 0x2A: RD_ACC; ROL; WB_ACC;  CYCLES(2); break;
		case 0x4A: RD_ACC; LSR; WB_ACC;  CYCLES(2); break;
		case 0x6A: RD_ACC; ROR; WB_ACC;  CYCLES(2); break;
		case 0x8A:         TXA;          CYCLES(2); break;
		case 0xAA:         TAX;          CYCLES(2); break;
		case 0xCA:         DEX;          CYCLES(2); break;
		case 0xEA:         NOP;          CYCLES(2); break;
	
		case 0x1A:         ILL;          CYCLES(2); break;
		case 0x3A:         ILL;          CYCLES(2); break;
		case 0x5A:         ILL;          CYCLES(2); break;
		case 0x7A:         ILL;          CYCLES(2); break;
		case 0x9A:         TXS;          CYCLES(2); break;
		case 0xBA:         TSX;          CYCLES(2); break;
		case 0xDA:         ILL;          CYCLES(2); break;
		case 0xFA:         ILL;          CYCLES(2); break;
		
		case 0x0B:         ILL;          CYCLES(2); break;
		case 0x2B:         ILL;          CYCLES(2); break;
		case 0x4B:         ILL;          CYCLES(2); break;
		case 0x6B:         ILL;          CYCLES(2); break;
		case 0x8B:         ILL;          CYCLES(2); break;
		case 0xAB:         ILL;          CYCLES(2); break;
		case 0xCB:         ILL;          CYCLES(2); break;
		case 0xEB:         ILL;          CYCLES(2); break;
	
		case 0x1B:         ILL;          CYCLES(2); break;
		case 0x3B:         ILL;          CYCLES(2); break;
		case 0x5B:         ILL;          CYCLES(2); break;
		case 0x7B:         ILL;          CYCLES(2); break;
		case 0x9B:         ILL;          CYCLES(2); break;
		case 0xBB:         ILL;          CYCLES(2); break;
		case 0xDB:         ILL;          CYCLES(2); break;
		case 0xFB:         ILL;          CYCLES(2); break;
	
		case 0x0C:         ILL;          CYCLES(2); break;
		case 0x2C: RD_ABS; BIT;          CYCLES(4); break;
		case 0x4C: EA_ABS; JMP;          CYCLES(3); break;
		case 0x6C: EA_IND; JMP;          CYCLES(5); break;
		case 0x8C:         STY; WR_ABS;  CYCLES(4); break;
		case 0xAC: RD_ABS; LDY;          CYCLES(4); break;
		case 0xCC: RD_ABS; CPY;          CYCLES(4); break;
		case 0xEC: RD_ABS; CPX;          CYCLES(4); break;

		case 0x1C:         ILL;          CYCLES(2); break;
		case 0x3C:         ILL;          CYCLES(2); break;
		case 0x5C:         ILL;          CYCLES(2); break;
		case 0x7C:         ILL;          CYCLES(2); break;
		case 0x9C:         ILL;          CYCLES(2); break;
		case 0xBC: RD_ABX; LDY;          CYCLES(4); break;
		case 0xDC:         ILL;          CYCLES(2); break;
		case 0xFC:         ILL;          CYCLES(2); break;
		
		case 0x0D: RD_ABS; ORA;          CYCLES(4); break;
		case 0x2D: RD_ABS; AND;          CYCLES(4); break;
		case 0x4D: RD_ABS; EOR;          CYCLES(4); break;
		case 0x6D: RD_ABS; ADC;          CYCLES(4); break;
		case 0x8D:         STA; WR_ABS;  CYCLES(4); break;
		case 0xAD: RD_ABS; LDA;          CYCLES(4); break;
		case 0xCD: RD_ABS; CMP;          CYCLES(4); break;
		case 0xED: RD_ABS; SBC;          CYCLES(4); break;
	
		case 0x1D: RD_ABX; ORA;          CYCLES(4); break;
		case 0x3D: RD_ABX; AND;          CYCLES(4); break;
		case 0x5D: RD_ABX; EOR;          CYCLES(4); break;
		case 0x7D: RD_ABX; ADC;          CYCLES(4); break;
		case 0x9D:         STA; WR_ABX;  CYCLES(5); break;
		case 0xBD: RD_ABX; LDA;          CYCLES(4); break;
		case 0xDD: RD_ABX; CMP;          CYCLES(4); break;
		case 0xFD: RD_ABX; SBC;          CYCLES(4); break;
	
		case 0x0E: RD_ABS; ASL; WB_EA;   CYCLES(6); break;
		case 0x2E: RD_ABS; ROL; WB_EA;   CYCLES(6); break;
		case 0x4E: RD_ABS; LSR; WB_EA;   CYCLES(6); break;
		case 0x6E: RD_ABS; ROR; WB_EA;   CYCLES(6); break;
		case 0x8E:         STX; WR_ABS;  CYCLES(5); break;
		case 0xAE: RD_ABS; LDX;          CYCLES(4); break;
		case 0xCE: RD_ABS; DEC; WB_EA;   CYCLES(6); break;
		case 0xEE: RD_ABS; INC; WB_EA;   CYCLES(6); break;
	
		case 0x1E: RD_ABX; ASL; WB_EA;   CYCLES(7); break;
		case 0x3E: RD_ABX; ROL; WB_EA;   CYCLES(7); break;
		case 0x5E: RD_ABX; LSR; WB_EA;   CYCLES(7); break;
		case 0x7E: RD_ABX; ROR; WB_EA;   CYCLES(7); break;
		case 0x9E:         ILL;          CYCLES(2); break;
		case 0xBE: RD_ABY; LDX;          CYCLES(4); break;
		case 0xDE: RD_ABX; DEC; WB_EA;   CYCLES(7); break;
		case 0xFE: RD_ABX; INC; WB_EA;   CYCLES(7); break;
	
		case 0x0F:         ILL;          CYCLES(2); break;
		case 0x2F:         ILL;          CYCLES(2); break;
		case 0x4F:         ILL;          CYCLES(2); break;
		case 0x6F:         ILL;          CYCLES(2); break;
		case 0x8F:         ILL;          CYCLES(2); break;
		case 0xAF:         ILL;          CYCLES(2); break;
		case 0xCF:         ILL;          CYCLES(2); break;
		case 0xEF:         ILL;          CYCLES(2); break;
		
		case 0x1F:         ILL;          CYCLES(2); break;
		case 0x3F:         ILL;          CYCLES(2); break;
		case 0x5F:         ILL;          CYCLES(2); break;
		case 0x7F:         ILL;          CYCLES(2); break;
		case 0x9F:         ILL;          CYCLES(2); break;
		case 0xBF:         ILL;          CYCLES(2); break;
		case 0xDF:         ILL;          CYCLES(2); break;
		case 0xFF:         ILL;          CYCLES(2); break;
	}
}	


void m6502_ResetCPU(register M65_CPU *m65)
{
	SP = 0x01FF;
	P  = 0;
	A  = 0;
	X  = 0;
	Y  = 0;

	IRQ = 0;

	// Reset initial PC register from
	// RST vector address.
	PCL = RD_MEM(RST_ADR);
	PCH = RD_MEM(RST_ADR+1);
}

inline void m6502_TakeIRQ(register M65_CPU *m65)
{
#ifdef DEBUG
	uint16 oldPC = PC;
	uint8  oldP  = P;
#endif /* DEBUG */

	// Save current PC and P register.
	PUSH(PCH); PUSH(PCL);
	PUSH(P & ~SR_B);
	P |= SR_I;

	// Load IRQ address
	PCL = RD_MEM(IRQ_ADR);
	PCH = RD_MEM(IRQ_ADR+1);

	// Count CPU cycles and reset IRQ line
	CYCLES(7);
	IRQ = 0;

#ifdef DEBUG
	if (dbg_Check(DBG_INTERRUPT))
		dbg_Printf("%s: IRQ - Old PC=%04X P=%02X => New PC=%04X P=%02X\n",
			m65->Unit.devNane, oldPC, oldP, PC, P);
#endif /* DEBUG */
}

inline void m6502_TakeNMI(register M65_CPU *m65)
{
#ifdef DEBUG
	uint16 oldPC = PC;
	uint8  oldP  = P;
#endif /* DEBUG */

	// Save current PC and P register.
	PUSH(PCH); PUSH(PCL);
	PUSH(P & ~SR_B);
	P |= SR_I;

	// Load IRQ address
	PCL = RD_MEM(IRQ_ADR);
	PCH = RD_MEM(IRQ_ADR+1);

	// Count CPU cycles
	CYCLES(7);

#ifdef DEBUG
	if (dbg_Check(DBG_INTERRUPT))
		dbg_Printf("%s: NMI - Old PC=%04X P=%02X => New PC=%04X P=%02X\n",
			m65->Unit.devNane, oldPC, oldP, PC, P);
#endif /* DEBUG */
}

void m6502_Execute(MAP_DEVICE *map)
{
	register M65_CPU *m65 = (M65_CPU *)map->Device;
	uint8 opCode;

	while (1) {
		if (IRQ && ((P & SR_I) == 0))
			m6502_TakeIRQ(m65);
#ifdef DEBUG
		if (dbg_Check(DBG_TRACE))
			m65_Disasm(m65);
#endif /* DEBUG */
		IPS++;
		m6502_ExecuteOpcode(m65, RD_OP);
	}
}
